"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
class nftBid extends sequelize_1.Model {
    static initModel(sequelize) {
        return nftBid.init({
            id: {
                type: sequelize_1.DataTypes.UUID,
                defaultValue: sequelize_1.DataTypes.UUIDV4,
                primaryKey: true,
                allowNull: false,
            },
            listingId: {
                type: sequelize_1.DataTypes.UUID,
                allowNull: false,
                validate: {
                    notNull: { msg: "listingId: Listing ID cannot be null" },
                    isUUID: { args: 4, msg: "listingId: Listing ID must be a valid UUID" },
                },
            },
            bidderId: {
                type: sequelize_1.DataTypes.UUID,
                allowNull: false,
                validate: {
                    notNull: { msg: "bidderId: Bidder ID cannot be null" },
                    isUUID: { args: 4, msg: "bidderId: Bidder ID must be a valid UUID" },
                },
            },
            amount: {
                type: sequelize_1.DataTypes.DECIMAL(36, 18),
                allowNull: false,
                validate: {
                    min: { args: [0], msg: "amount: Amount must be positive" },
                },
            },
            currency: {
                type: sequelize_1.DataTypes.STRING(10),
                allowNull: false,
                defaultValue: "ETH",
                validate: {
                    notEmpty: { msg: "currency: Currency must not be empty" },
                },
            },
            expiresAt: {
                type: sequelize_1.DataTypes.DATE,
                allowNull: true,
            },
            status: {
                type: sequelize_1.DataTypes.ENUM("ACTIVE", "ACCEPTED", "REJECTED", "EXPIRED", "CANCELLED"),
                allowNull: false,
                defaultValue: "ACTIVE",
                validate: {
                    isIn: {
                        args: [["ACTIVE", "ACCEPTED", "REJECTED", "EXPIRED", "CANCELLED"]],
                        msg: "status: Status must be one of 'ACTIVE', 'ACCEPTED', 'REJECTED', 'EXPIRED', or 'CANCELLED'",
                    },
                },
            },
            metadata: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: true,
                get() {
                    const value = this.getDataValue("metadata");
                    return value ? JSON.parse(value) : null;
                },
                set(value) {
                    this.setDataValue("metadata", JSON.stringify(value));
                },
            },
        }, {
            sequelize,
            modelName: "nftBid",
            tableName: "nft_bid",
            timestamps: true,
            paranoid: true,
            indexes: [
                {
                    name: "PRIMARY",
                    unique: true,
                    using: "BTREE",
                    fields: [{ name: "id" }],
                },
                {
                    name: "nftBidListingIdx",
                    using: "BTREE",
                    fields: [{ name: "listingId" }],
                },
                {
                    name: "nftBidBidderIdx",
                    using: "BTREE",
                    fields: [{ name: "bidderId" }],
                },
                {
                    name: "nftBidStatusIdx",
                    using: "BTREE",
                    fields: [{ name: "status" }],
                },
                {
                    name: "nftBidAmountIdx",
                    using: "BTREE",
                    fields: [{ name: "amount" }],
                },
                {
                    name: "nftBidExpiresAtIdx",
                    using: "BTREE",
                    fields: [{ name: "expiresAt" }],
                },
            ],
        });
    }
    static associate(models) {
        nftBid.belongsTo(models.nftListing, {
            as: "listing",
            foreignKey: "listingId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        nftBid.belongsTo(models.user, {
            as: "bidder",
            foreignKey: "bidderId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
    }
}
exports.default = nftBid;
