"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
class nftListing extends sequelize_1.Model {
    static initModel(sequelize) {
        return nftListing.init({
            id: {
                type: sequelize_1.DataTypes.UUID,
                defaultValue: sequelize_1.DataTypes.UUIDV4,
                primaryKey: true,
                allowNull: false,
            },
            tokenId: {
                type: sequelize_1.DataTypes.UUID,
                allowNull: false,
                validate: {
                    notNull: { msg: "tokenId: Token ID cannot be null" },
                    isUUID: { args: 4, msg: "tokenId: Token ID must be a valid UUID" },
                },
            },
            sellerId: {
                type: sequelize_1.DataTypes.UUID,
                allowNull: false,
                validate: {
                    notNull: { msg: "sellerId: Seller ID cannot be null" },
                    isUUID: { args: 4, msg: "sellerId: Seller ID must be a valid UUID" },
                },
            },
            type: {
                type: sequelize_1.DataTypes.ENUM("FIXED_PRICE", "AUCTION", "BUNDLE"),
                allowNull: false,
                defaultValue: "FIXED_PRICE",
                validate: {
                    isIn: {
                        args: [["FIXED_PRICE", "AUCTION", "BUNDLE"]],
                        msg: "type: Type must be one of 'FIXED_PRICE', 'AUCTION', or 'BUNDLE'",
                    },
                },
            },
            price: {
                type: sequelize_1.DataTypes.DECIMAL(36, 18),
                allowNull: true,
                validate: {
                    min: { args: [0], msg: "price: Price must be non-negative" },
                },
            },
            currency: {
                type: sequelize_1.DataTypes.STRING(10),
                allowNull: false,
                defaultValue: "ETH",
                validate: {
                    notEmpty: { msg: "currency: Currency must not be empty" },
                },
            },
            reservePrice: {
                type: sequelize_1.DataTypes.DECIMAL(36, 18),
                allowNull: true,
                validate: {
                    min: { args: [0], msg: "reservePrice: Reserve price must be non-negative" },
                },
            },
            buyNowPrice: {
                type: sequelize_1.DataTypes.DECIMAL(36, 18),
                allowNull: true,
                validate: {
                    min: { args: [0], msg: "buyNowPrice: Buy now price must be non-negative" },
                },
            },
            startTime: {
                type: sequelize_1.DataTypes.DATE,
                allowNull: true,
            },
            endTime: {
                type: sequelize_1.DataTypes.DATE,
                allowNull: true,
            },
            status: {
                type: sequelize_1.DataTypes.ENUM("ACTIVE", "SOLD", "CANCELLED", "EXPIRED"),
                allowNull: false,
                defaultValue: "ACTIVE",
                validate: {
                    isIn: {
                        args: [["ACTIVE", "SOLD", "CANCELLED", "EXPIRED"]],
                        msg: "status: Status must be one of 'ACTIVE', 'SOLD', 'CANCELLED', or 'EXPIRED'",
                    },
                },
            },
            views: {
                type: sequelize_1.DataTypes.INTEGER,
                allowNull: false,
                defaultValue: 0,
                validate: {
                    min: { args: [0], msg: "views: Views must be non-negative" },
                },
            },
            likes: {
                type: sequelize_1.DataTypes.INTEGER,
                allowNull: false,
                defaultValue: 0,
                validate: {
                    min: { args: [0], msg: "likes: Likes must be non-negative" },
                },
            },
            metadata: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: true,
                get() {
                    const value = this.getDataValue("metadata");
                    return value ? JSON.parse(value) : null;
                },
                set(value) {
                    this.setDataValue("metadata", JSON.stringify(value));
                },
            },
        }, {
            sequelize,
            modelName: "nftListing",
            tableName: "nft_listing",
            timestamps: true,
            paranoid: true,
            indexes: [
                {
                    name: "PRIMARY",
                    unique: true,
                    using: "BTREE",
                    fields: [{ name: "id" }],
                },
                {
                    name: "nftListingTokenIdx",
                    using: "BTREE",
                    fields: [{ name: "tokenId" }],
                },
                {
                    name: "nftListingSellerIdx",
                    using: "BTREE",
                    fields: [{ name: "sellerId" }],
                },
                {
                    name: "nftListingStatusIdx",
                    using: "BTREE",
                    fields: [{ name: "status" }],
                },
                {
                    name: "nftListingTypeIdx",
                    using: "BTREE",
                    fields: [{ name: "type" }],
                },
                {
                    name: "nftListingPriceIdx",
                    using: "BTREE",
                    fields: [{ name: "price" }],
                },
            ],
        });
    }
    static associate(models) {
        nftListing.belongsTo(models.nftToken, {
            as: "token",
            foreignKey: "tokenId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        nftListing.belongsTo(models.user, {
            as: "seller",
            foreignKey: "sellerId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        nftListing.hasMany(models.nftBid, {
            as: "bids",
            foreignKey: "listingId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        nftListing.hasMany(models.nftOffer, {
            as: "offers",
            foreignKey: "listingId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        nftListing.hasMany(models.nftActivity, {
            as: "activities",
            foreignKey: "listingId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
    }
}
exports.default = nftListing;
