"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
class p2pOffer extends sequelize_1.Model {
    static initModel(sequelize) {
        return p2pOffer.init({
            id: {
                type: sequelize_1.DataTypes.UUID,
                defaultValue: sequelize_1.DataTypes.UUIDV4,
                primaryKey: true,
                allowNull: false,
            },
            userId: {
                type: sequelize_1.DataTypes.UUID,
                allowNull: false,
                validate: { isUUID: { args: 4, msg: "userId must be a valid UUID" } },
            },
            type: {
                type: sequelize_1.DataTypes.ENUM("BUY", "SELL"),
                allowNull: false,
                validate: {
                    isIn: { args: [["BUY", "SELL"]], msg: "Invalid trade type" },
                },
            },
            currency: {
                type: sequelize_1.DataTypes.STRING(50),
                allowNull: false,
                validate: { notEmpty: { msg: "currency must not be empty" } },
            },
            walletType: {
                type: sequelize_1.DataTypes.ENUM("FIAT", "SPOT", "ECO"),
                allowNull: false,
                validate: {
                    isIn: {
                        args: [["FIAT", "SPOT", "ECO"]],
                        msg: "Invalid wallet type",
                    },
                },
            },
            amountConfig: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: false,
            },
            priceConfig: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: false,
            },
            tradeSettings: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: false,
            },
            locationSettings: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: true,
            },
            userRequirements: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: true,
            },
            status: {
                type: sequelize_1.DataTypes.ENUM("DRAFT", "PENDING_APPROVAL", "ACTIVE", "PAUSED", "COMPLETED", "CANCELLED", "REJECTED", "EXPIRED"),
                allowNull: false,
                defaultValue: "DRAFT",
            },
            views: {
                type: sequelize_1.DataTypes.INTEGER,
                allowNull: false,
                defaultValue: 0,
            },
            systemTags: {
                type: sequelize_1.DataTypes.JSON,
                allowNull: true,
            },
            adminNotes: {
                type: sequelize_1.DataTypes.TEXT,
                allowNull: true,
            },
        }, {
            sequelize,
            modelName: "p2pOffer",
            tableName: "p2p_offers",
            timestamps: true,
            paranoid: true,
        });
    }
    static associate(models) {
        p2pOffer.belongsTo(models.user, {
            as: "user",
            foreignKey: "userId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        p2pOffer.belongsToMany(models.p2pPaymentMethod, {
            through: "p2p_offer_payment_method",
            as: "paymentMethods",
            foreignKey: "offerId",
            otherKey: "paymentMethodId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        p2pOffer.hasOne(models.p2pOfferFlag, {
            as: "flag",
            foreignKey: "offerId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
        p2pOffer.hasMany(models.p2pTrade, {
            as: "trades",
            foreignKey: "offerId",
            onDelete: "CASCADE",
            onUpdate: "CASCADE",
        });
    }
}
exports.default = p2pOffer;
