"use strict";
// /api/mlm/referrals/store.post.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
const db_1 = require("@b/db");
const cache_1 = require("@b/utils/cache");
const affiliate_1 = require("@b/utils/affiliate");
exports.metadata = {
    summary: "Stores a new MLM Referral",
    operationId: "storeMlmReferral",
    tags: ["Admin", "MLM", "Referrals"],
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: utils_1.mlmReferralUpdateSchema,
            },
        },
    },
    responses: (0, query_1.storeRecordResponses)(utils_1.mlmReferralStoreSchema, "MLM Referral"),
    requiresAuth: true,
    permission: "create.affiliate.referral",
};
exports.default = async (data) => {
    const { body } = data;
    const { status, referrerId, referredId } = body;
    if (referrerId === referredId)
        throw new Error("Referrer and referred user cannot be the same");
    const referrer = await db_1.models.user.findOne({ where: { id: referrerId } });
    if (!referrer)
        throw new Error("Referrer not found");
    const referred = await db_1.models.user.findOne({ where: { id: referredId } });
    if (!referred)
        throw new Error("Referred user not found");
    // Create the referral record.
    const newReferral = await (0, query_1.storeRecord)({
        model: "mlmReferral",
        data: {
            status,
            referrerId,
            referredId,
        },
    });
    const cacheManager = cache_1.CacheManager.getInstance();
    const settings = await cacheManager.getSettings();
    const mlmSystem = settings.has("mlmSystem")
        ? settings.get("mlmSystem")
        : null;
    // For DIRECT system, skip node creation.
    if (mlmSystem === "DIRECT") {
        return newReferral;
    }
    else if (mlmSystem === "BINARY") {
        await (0, affiliate_1.handleBinaryMlmReferralRegister)(referrerId, newReferral, db_1.models.mlmBinaryNode);
    }
    else if (mlmSystem === "UNILEVEL") {
        await (0, affiliate_1.handleUnilevelMlmReferralRegister)(referrerId, newReferral, db_1.models.mlmUnilevelNode);
    }
    return newReferral;
};
