"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const error_1 = require("@b/utils/error");
const db_1 = require("@b/db");
exports.metadata = {
    summary: "Retrieves ecosystem blockchain options",
    description: "This endpoint retrieves a list of blockchain options for the ecosystem. It includes a static list of blockchains, and conditionally adds Solana and Mo Chain if they are enabled.",
    operationId: "getEcosystemBlockchainOptions",
    tags: ["Ecosystem", "Blockchain"],
    requiresAuth: true,
    responses: {
        200: {
            description: "Blockchain options retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                id: { type: "string" },
                                name: { type: "string" },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("EcosystemBlockchain"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id))
        throw (0, error_1.createError)(401, "Unauthorized");
    try {
        // Retrieve blockchain statuses for SOL and MO.
        const chains = await db_1.models.ecosystemBlockchain.findAll({
            where: { chain: ["SOL", "MO"] },
        });
        const solanaBlockchain = chains.find((c) => c.chain === "SOL" && c.status);
        const moBlockchain = chains.find((c) => c.chain === "MO" && c.status);
        // Base list of static blockchain options.
        const blockchainOptions = [
            { id: "ARBITRUM", name: "Arbitrum (ARB)" },
            { id: "BASE", name: "Base (BASE)" },
            { id: "BSC", name: "Binance Smart Chain (BSC)" },
            { id: "CELO", name: "Celo (CELO)" },
            { id: "ETH", name: "Ethereum (ETH)" },
            { id: "FTM", name: "Fantom (FTM)" },
            { id: "OPTIMISM", name: "Optimism (OVM)" },
            { id: "POLYGON", name: "Polygon (MATIC)" },
            { id: "RSK", name: "RSK (RSK)" },
            ...(solanaBlockchain ? [{ id: "SOL", name: "Solana (SOL)" }] : []),
            ...(moBlockchain ? [{ id: "MO", name: "Mo Chain (MO)" }] : []),
        ];
        return blockchainOptions;
    }
    catch (error) {
        throw (0, error_1.createError)(500, "An error occurred while fetching blockchain options");
    }
};
