"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const queries_1 = require("@b/api/(ext)/ecosystem/utils/scylla/queries");
const query_1 = require("@b/utils/query");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Bulk deletes ecosystem markets by IDs",
    operationId: "bulkDeleteEcosystemMarkets",
    tags: ["Admin", "Ecosystem", "Market"],
    parameters: (0, query_1.commonBulkDeleteParams)("Ecosystem Markets"),
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        ids: {
                            type: "array",
                            items: { type: "string" },
                            description: "Array of ecosystem market IDs to delete",
                        },
                    },
                    required: ["ids"],
                },
            },
        },
    },
    responses: (0, query_1.commonBulkDeleteResponses)("Ecosystem Markets"),
    requiresAuth: true,
    permission: "delete.ecosystem.market",
};
exports.default = async (data) => {
    const { body, query } = data;
    const { ids } = body;
    // Validate payload
    if (!ids || !Array.isArray(ids) || ids.length === 0) {
        throw (0, error_1.createError)(400, "No market IDs provided");
    }
    // Find all markets matching the provided IDs, retrieving their currency fields
    const markets = await db_1.models.ecosystemMarket.findAll({
        where: { id: ids },
        attributes: ["currency"],
        paranoid: false,
    });
    console.log("🚀 ~ markets:", markets);
    if (!markets.length) {
        throw (0, error_1.createError)(404, "No matching markets found for provided IDs");
    }
    // Define a post-delete action to remove all market data for each market using its currency.
    const postDelete = async () => {
        for (const market of markets) {
            await (0, queries_1.deleteAllMarketData)(market.currency);
        }
    };
    return (0, query_1.handleBulkDelete)({
        model: "ecosystemMarket",
        ids: ids,
        query: { ...query, force: true },
        postDelete,
    });
};
