"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("@b/api/admin/finance/exchange/market/utils");
const db_1 = require("@b/db"); // Adjust path as needed
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Stores a new Ecosystem Market",
    operationId: "storeEcosystemMarket",
    tags: ["Admin", "Ecosystem Markets"],
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: utils_1.MarketUpdateSchema,
            },
        },
    },
    responses: (0, query_1.storeRecordResponses)(utils_1.MarketStoreSchema, "Ecosystem Market"),
    requiresAuth: true,
    permission: "create.ecosystem.market",
};
exports.default = async (data) => {
    const { body } = data;
    const { currency, pair, isTrending, isHot, metadata } = body;
    // 1) Find the currency token by ID
    const currencyToken = await db_1.models.ecosystemToken.findOne({
        where: { id: currency, status: true },
    });
    if (!currencyToken) {
        throw (0, error_1.createError)(404, "Currency token not found or inactive");
    }
    // 2) Find the pair token by ID
    const pairToken = await db_1.models.ecosystemToken.findOne({
        where: { id: pair, status: true },
    });
    if (!pairToken) {
        throw (0, error_1.createError)(404, "Pair token not found or inactive");
    }
    // 2.1) Check if a market with the same currency and pair already exists.
    //     (Assuming a unique constraint on the combination of currency and pair.)
    const existingMarket = await db_1.models.ecosystemMarket.findOne({
        where: {
            currency: currencyToken.currency, // or use currencyToken.symbol if preferred
            pair: pairToken.currency,
        },
    });
    if (existingMarket) {
        throw (0, error_1.createError)(409, "Ecosystem market with the given currency and pair already exists.");
    }
    // 3) Store the new market
    try {
        return await (0, query_1.storeRecord)({
            model: "ecosystemMarket",
            data: {
                currency: currencyToken.currency, // or currencyToken.symbol if preferred
                pair: pairToken.currency,
                isTrending,
                isHot,
                metadata,
                status: true,
            },
        });
    }
    catch (error) {
        // If the error is due to a unique constraint violation, throw a 409 error.
        if (error.name === "SequelizeUniqueConstraintError") {
            throw (0, error_1.createError)(409, "Ecosystem market already exists.");
        }
        // Otherwise, rethrow the error.
        throw error;
    }
};
