"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const error_1 = require("@b/utils/error");
const db_1 = require("@b/db");
exports.metadata = {
    summary: "Retrieves ecosystem token options",
    description: "This endpoint retrieves active ecosystem tokens and formats them as selectable options.",
    operationId: "getEcosystemTokenOptions",
    tags: ["Ecosystem", "Token"],
    requiresAuth: true,
    responses: {
        200: {
            description: "Ecosystem tokens retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                id: { type: "string" },
                                name: { type: "string" },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("EcosystemToken"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id))
        throw (0, error_1.createError)(401, "Unauthorized");
    try {
        const tokens = await db_1.models.ecosystemToken.findAll({
            where: { status: true },
        });
        // Deduplicate by the 'currency' field
        const seenSymbols = new Set();
        const deduplicated = [];
        for (const token of tokens) {
            // e.g. token.currency might be "MO" or "USDT"
            if (!seenSymbols.has(token.currency)) {
                seenSymbols.add(token.currency);
                deduplicated.push({
                    id: token.id,
                    name: `${token.currency} - ${token.name} (${token.chain})`,
                });
            }
        }
        return deduplicated;
    }
    catch (error) {
        throw (0, error_1.createError)(500, "An error occurred while fetching ecosystem tokens");
    }
};
