"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getEcosystemMasterWalletBalance = exports.createEVMWallet = exports.createAndEncryptWallet = exports.ecosystemMasterWalletStoreSchema = exports.ecosystemMasterWalletUpdateSchema = exports.baseEcosystemMasterWalletSchema = exports.ecosystemMasterWalletSchema = void 0;
exports.getAllMasterWallets = getAllMasterWallets;
exports.getMasterWalletById = getMasterWalletById;
exports.getMasterWallet = getMasterWallet;
exports.createMasterWallet = createMasterWallet;
exports.updateMasterWalletBalance = updateMasterWalletBalance;
exports.deployCustodialContract = deployCustodialContract;
const fs = __importStar(require("fs"));
const utxo_1 = require("@b/api/(ext)/ecosystem/utils/utxo");
const encrypt_1 = require("@b/utils/encrypt");
const schema_1 = require("@b/utils/schema");
const ethers_1 = require("ethers");
const redis_1 = require("@b/utils/redis");
const date_fns_1 = require("date-fns");
const gas_1 = require("@b/api/(ext)/ecosystem/utils/gas");
const db_1 = require("@b/db");
const provider_1 = require("@b/api/(ext)/ecosystem/utils/provider");
const chains_1 = require("@b/api/(ext)/ecosystem/utils/chains");
const smartContract_1 = require("@b/api/(ext)/ecosystem/utils/smartContract");
const wallet_1 = require("@b/api/(ext)/ecosystem/utils/wallet");
const safe_imports_1 = require("@b/utils/safe-imports");
const path_1 = __importDefault(require("path"));
// Fetch all master wallets
async function getAllMasterWallets() {
    return db_1.models.ecosystemMasterWallet.findAll({
        attributes: wallet_1.walletResponseAttributes,
    });
}
// Fetch a single master wallet by ID
async function getMasterWalletById(id) {
    return db_1.models.ecosystemMasterWallet.findOne({
        where: { id },
        attributes: wallet_1.walletResponseAttributes,
    });
}
// Fetch a single master wallet by UUID (no select constraint)
async function getMasterWallet(id) {
    return db_1.models.ecosystemMasterWallet.findOne({
        where: { id },
    });
}
// Create a new master wallet
async function createMasterWallet(walletData, currency) {
    const wallet = await db_1.models.ecosystemMasterWallet.create({
        currency,
        chain: walletData.chain,
        address: walletData.address,
        data: walletData.data,
        status: true,
    });
    return wallet;
}
// Update master wallet balance
async function updateMasterWalletBalance(id, balance) {
    await db_1.models.ecosystemMasterWallet.update({
        balance,
    }, {
        where: { id },
    });
    return getMasterWalletById(id);
}
const id = (0, schema_1.baseStringSchema)("ID of the ecosystem master wallet");
const chain = (0, schema_1.baseStringSchema)("Blockchain chain associated with the master wallet", 255);
const currency = (0, schema_1.baseStringSchema)("Currency used in the master wallet", 255);
const address = (0, schema_1.baseStringSchema)("Address of the master wallet", 255);
const balance = (0, schema_1.baseNumberSchema)("Balance of the master wallet");
const data = (0, schema_1.baseStringSchema)("Additional data associated with the master wallet", 1000, 0, true);
const status = (0, schema_1.baseEnumSchema)("Operational status of the master wallet", [
    "ACTIVE",
    "INACTIVE",
]);
const lastIndex = (0, schema_1.baseNumberSchema)("Last index used for generating wallet address");
exports.ecosystemMasterWalletSchema = {
    id,
    chain,
    currency,
    address,
    balance,
    data,
    status,
    lastIndex,
};
exports.baseEcosystemMasterWalletSchema = {
    id,
    chain,
    currency,
    address,
    balance,
    data,
    status,
    lastIndex,
};
exports.ecosystemMasterWalletUpdateSchema = {
    type: "object",
    properties: {
        chain,
        currency,
        address,
        balance,
        data,
        status,
        lastIndex,
    },
    required: ["chain", "currency", "address", "status", "lastIndex"],
};
exports.ecosystemMasterWalletStoreSchema = {
    description: `Master wallet created or updated successfully`,
    content: {
        "application/json": {
            schema: {
                type: "object",
                properties: exports.baseEcosystemMasterWalletSchema,
            },
        },
    },
};
const createAndEncryptWallet = async (chain) => {
    let wallet;
    if (["BTC", "LTC", "DOGE", "DASH"].includes(chain)) {
        // Generate a UTXO wallet
        wallet = (0, utxo_1.createUTXOWallet)(chain);
    }
    else if (chain === "SOL") {
        const SolanaService = await (0, safe_imports_1.getSolanaService)();
        if (!SolanaService) {
            throw new Error("Solana service not available");
        }
        const solanaService = await SolanaService.getInstance();
        wallet = solanaService.createWallet();
    }
    else if (chain === "TRON") {
        const TronService = await (0, safe_imports_1.getTronService)();
        if (!TronService) {
            throw new Error("Tron service not available");
        }
        const tronService = await TronService.getInstance();
        wallet = tronService.createWallet();
    }
    else if (chain === "XMR") {
        const MoneroService = await (0, safe_imports_1.getMoneroService)();
        if (!MoneroService) {
            throw new Error("Monero service not available");
        }
        const moneroService = await MoneroService.getInstance();
        wallet = await moneroService.createWallet("master_wallet");
    }
    else if (chain === "TON") {
        const TonService = await (0, safe_imports_1.getTonService)();
        if (!TonService) {
            throw new Error("TON service not available");
        }
        const tonService = await TonService.getInstance();
        wallet = await tonService.createWallet();
    }
    else {
        // Generate an EVM wallet
        wallet = (0, exports.createEVMWallet)();
    }
    // Define the directory and file path with fallback for different environments
    const possibleWalletDirs = [
        path_1.default.resolve(process.cwd(), "backend", "ecosystem", "wallets"), // Production path (PRIORITY)
        path_1.default.resolve(__dirname, "../../../../../ecosystem", "wallets"), // Development relative path
        path_1.default.resolve(process.cwd(), "ecosystem", "wallets"), // Alternative if running from backend dir
        path_1.default.resolve(__dirname, "../../../../ecosystem", "wallets"), // Another relative fallback
    ];
    let walletDir = possibleWalletDirs[0];
    // Try to find an existing directory or use the first path for creation
    for (const possibleDir of possibleWalletDirs) {
        const parentDir = path_1.default.dirname(possibleDir);
        if (fs.existsSync(parentDir)) {
            walletDir = possibleDir;
            console.log(`Using wallet directory: ${walletDir}`);
            break;
        }
    }
    const walletFilePath = `${walletDir}/${chain}.json`;
    // Check if directory exists, if not create it
    if (!fs.existsSync(walletDir)) {
        fs.mkdirSync(walletDir, { recursive: true });
    }
    await fs.writeFileSync(walletFilePath, JSON.stringify(wallet), "utf8");
    // Encrypt all the wallet details
    const data = (0, encrypt_1.encrypt)(JSON.stringify(wallet.data));
    return {
        address: wallet.address,
        chain,
        data,
    };
};
exports.createAndEncryptWallet = createAndEncryptWallet;
const createEVMWallet = () => {
    // Generate a random wallet
    const wallet = ethers_1.ethers.Wallet.createRandom();
    if (!wallet.mnemonic) {
        throw new Error("Mnemonic not found");
    }
    // Derive the HDNode from the wallet's mnemonic
    const hdNode = ethers_1.ethers.HDNodeWallet.fromPhrase(wallet.mnemonic.phrase);
    if (!hdNode) {
        throw new Error("HDNode not found");
    }
    const xprv = hdNode.extendedKey;
    const xpub = hdNode.neuter().extendedKey;
    if (!hdNode.mnemonic) {
        throw new Error("Mnemonic not found");
    }
    const mnemonic = hdNode.mnemonic.phrase;
    const address = hdNode.address;
    const publicKey = hdNode.publicKey;
    const privateKey = hdNode.privateKey;
    const path = hdNode.path;
    const chainCode = hdNode.chainCode;
    return {
        address,
        data: {
            mnemonic,
            publicKey,
            privateKey,
            xprv,
            xpub,
            chainCode,
            path,
        },
    };
};
exports.createEVMWallet = createEVMWallet;
const getEcosystemMasterWalletBalance = async (wallet) => {
    var _a, _b, _c;
    try {
        const cacheKey = `wallet:${wallet.id}:balance`;
        const redis = redis_1.RedisSingleton.getInstance();
        let cachedBalanceData = await redis.get(cacheKey);
        if (cachedBalanceData) {
            if (typeof cachedBalanceData !== "object") {
                cachedBalanceData = JSON.parse(cachedBalanceData);
            }
            const now = new Date();
            const lastUpdated = new Date(cachedBalanceData.timestamp);
            // Cache for 1 minute instead of 5 to allow more frequent updates
            if ((0, date_fns_1.differenceInMinutes)(now, lastUpdated) < 1) {
                return;
            }
        }
        let formattedBalance;
        if (["BTC", "LTC", "DOGE", "DASH"].includes(wallet.chain)) {
            formattedBalance = await (0, utxo_1.fetchUTXOWalletBalance)(wallet.chain, wallet.address);
        }
        else if (wallet.chain === "SOL") {
            const SolanaService = await (0, safe_imports_1.getSolanaService)();
            if (!SolanaService) {
                throw new Error("Solana service not available");
            }
            const solanaService = await SolanaService.getInstance();
            formattedBalance = await solanaService.getBalance(wallet.address);
        }
        else if (wallet.chain === "TRON") {
            const TronService = await (0, safe_imports_1.getTronService)();
            if (!TronService) {
                throw new Error("Tron service not available");
            }
            const tronService = await TronService.getInstance();
            formattedBalance = await tronService.getBalance(wallet.address);
        }
        else if (wallet.chain === "XMR") {
            const MoneroService = await (0, safe_imports_1.getMoneroService)();
            if (!MoneroService) {
                console.log(`[${wallet.chain}] Monero service not available - skipping balance fetch`);
                return;
            }
            try {
                const moneroService = await MoneroService.getInstance();
                formattedBalance = await moneroService.getBalance("master_wallet");
            }
            catch (xmrError) {
                // Handle XMR-specific errors (chain not active, daemon not synchronized, etc.)
                if (((_a = xmrError.message) === null || _a === void 0 ? void 0 : _a.includes("not active")) || ((_b = xmrError.message) === null || _b === void 0 ? void 0 : _b.includes("not synchronized"))) {
                    console.log(`[${wallet.chain}] ${xmrError.message} - skipping balance fetch`);
                }
                else {
                    console.log(`[${wallet.chain}] Error fetching balance: ${(_c = xmrError.message) === null || _c === void 0 ? void 0 : _c.substring(0, 100)}`);
                }
                return;
            }
        }
        else if (wallet.chain === "TON") {
            const TonService = await (0, safe_imports_1.getTonService)();
            if (!TonService) {
                throw new Error("TON service not available");
            }
            const tonService = await TonService.getInstance();
            formattedBalance = await tonService.getBalance(wallet.address);
        }
        else {
            try {
                const provider = await (0, provider_1.getProvider)(wallet.chain);
                const balance = await provider.getBalance(wallet.address);
                const decimals = chains_1.chainConfigs[wallet.chain].decimals;
                formattedBalance = ethers_1.ethers.formatUnits(balance.toString(), decimals);
            }
            catch (providerError) {
                // Silently skip provider errors (network issues, SSL certs, etc.)
                console.log(`[${wallet.chain}] Provider error - skipping balance fetch`);
                return;
            }
        }
        if (!formattedBalance || isNaN(parseFloat(formattedBalance))) {
            console.log(`Invalid formatted balance for ${wallet.chain} wallet: ${formattedBalance}`);
            return;
        }
        // Always update the database, even if balance is 0
        const balanceFloat = parseFloat(formattedBalance);
        await updateMasterWalletBalance(wallet.id, balanceFloat);
        // Cache the balance for 1 minute
        const cacheData = {
            balance: formattedBalance,
            timestamp: new Date().toISOString(),
        };
        await redis.setex(cacheKey, 60, JSON.stringify(cacheData));
    }
    catch (error) {
        console.error(`Failed to fetch ${wallet.chain} wallet balance: ${error.message}`);
    }
};
exports.getEcosystemMasterWalletBalance = getEcosystemMasterWalletBalance;
async function deployCustodialContract(masterWallet) {
    try {
        // Initialize Ethereum provider
        const provider = await (0, provider_1.getProvider)(masterWallet.chain);
        if (!provider) {
            throw new Error("Provider not initialized");
        }
        // Decrypt mnemonic
        let decryptedData;
        if (!masterWallet.data) {
            throw new Error("Mnemonic not found");
        }
        try {
            decryptedData = JSON.parse((0, encrypt_1.decrypt)(masterWallet.data));
        }
        catch (error) {
            throw new Error(`Failed to decrypt mnemonic: ${error.message}`);
        }
        if (!decryptedData || !decryptedData.privateKey) {
            throw new Error("Decrypted data or Mnemonic not found");
        }
        const { privateKey } = decryptedData;
        // Create a signer
        const signer = new ethers_1.ethers.Wallet(privateKey).connect(provider);
        const { abi, bytecode } = await (0, smartContract_1.getSmartContract)("wallet", "CustodialWalletERC20");
        if (!abi || !bytecode) {
            throw new Error("Smart contract ABI or Bytecode not found");
        }
        // Create Contract Factory
        const custodialWalletFactory = new ethers_1.ContractFactory(abi, bytecode, signer);
        // Fetch adjusted gas price
        const gasPrice = await (0, gas_1.getAdjustedGasPrice)(provider);
        // Deploy the contract with dynamic gas settings
        const custodialWalletContract = await custodialWalletFactory.deploy(masterWallet.address, {
            gasPrice: gasPrice,
        });
        // Wait for the contract to be deployed
        const response = await custodialWalletContract.waitForDeployment();
        return await response.getAddress();
    }
    catch (error) {
        throw new Error(error.message);
    }
}
