"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const deepseek_client_1 = require("@b/utils/ai/deepseek-client");
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Answer FAQ Question",
    description: "Generates an answer to a user question based on existing active FAQs using the DeepSeek API.",
    operationId: "aiAnswerQuestion",
    tags: ["FAQ", "Admin", "AI"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        question: { type: "string", description: "User question" },
                    },
                    required: ["question"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Question answered successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            data: {
                                type: "object",
                                properties: { answer: { type: "string" } },
                            },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
    permission: "create.faq",
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { question } = body;
    if (!question) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Question is required" });
    }
    try {
        // Retrieve existing active FAQs for context.
        const faqs = await db_1.models.faq.findAll({
            where: { status: true },
            attributes: ["question", "answer"],
            raw: true,
        });
        const answer = await deepseek_client_1.deepseekClient.answerQuestion(question, faqs);
        return answer;
    }
    catch (error) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: error instanceof Error ? error.message : "Failed to answer question",
        });
    }
};
