"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const deepseek_client_1 = require("@b/utils/ai/deepseek-client");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Generate FAQ using AI",
    description: "Generates a comprehensive FAQ based on a topic (and optional context) using the DeepSeek API.",
    operationId: "aiGenerateFAQ",
    tags: ["FAQ", "Admin", "AI"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        topic: { type: "string", description: "FAQ topic" },
                        context: {
                            type: "string",
                            description: "Optional additional context",
                        },
                    },
                    required: ["topic"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "FAQ generated successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            data: { type: "object", description: "Generated FAQ object" },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
    permission: "create.faq",
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { topic, context } = body;
    if (!topic) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Topic is required" });
    }
    try {
        const generatedFAQ = await deepseek_client_1.deepseekClient.generateFAQ(topic, context);
        return generatedFAQ;
    }
    catch (error) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: error instanceof Error ? error.message : "Failed to generate FAQ",
        });
    }
};
