"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const deepseek_client_1 = require("@b/utils/ai/deepseek-client");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Improve FAQ Answer",
    description: "Improves an existing FAQ answer to make it more comprehensive and clear using the DeepSeek API.",
    operationId: "aiImproveFAQ",
    tags: ["FAQ", "Admin", "AI"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        question: { type: "string", description: "FAQ question" },
                        answer: { type: "string", description: "Current FAQ answer" },
                    },
                    required: ["question", "answer"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "FAQ improved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            data: {
                                type: "object",
                                properties: { answer: { type: "string" } },
                            },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
    permission: "create.faq",
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { question, answer } = body;
    if (!question || !answer) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Missing required fields" });
    }
    try {
        const improvedAnswer = await deepseek_client_1.deepseekClient.improveFAQ(question, answer);
        return improvedAnswer;
    }
    catch (error) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: error instanceof Error ? error.message : "Failed to improve FAQ",
        });
    }
};
