"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const deepseek_client_1 = require("@b/utils/ai/deepseek-client");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Summarize FAQ Content",
    description: "Generates a concise summary of the provided FAQ content using the DeepSeek API.",
    operationId: "aiSummarizeFAQContent",
    tags: ["FAQ", "Admin", "AI"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        content: {
                            type: "string",
                            description: "FAQ content to summarize",
                        },
                    },
                    required: ["content"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Content summarized successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            data: {
                                type: "object",
                                properties: { summary: { type: "string" } },
                            },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
    permission: "create.faq",
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { content } = body;
    if (!content) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Content is required" });
    }
    try {
        const summary = await deepseek_client_1.deepseekClient.summarizeFAQ(content);
        return summary;
    }
    catch (error) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: error instanceof Error ? error.message : "Failed to summarize content",
        });
    }
};
