"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
const db_1 = require("@b/db"); // Adjust path as needed
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Stores a new Futures Market",
    operationId: "storeFuturesMarket",
    tags: ["Admin", "Futures Markets"],
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: utils_1.FuturesMarketUpdateSchema,
            },
        },
    },
    responses: (0, query_1.storeRecordResponses)(utils_1.FuturesMarketStoreSchema, "Futures Market"),
    requiresAuth: true,
    permission: "create.futures.market",
};
exports.default = async (data) => {
    const { body } = data;
    const { currency, pair, isTrending, isHot, metadata } = body;
    // 1) Find the currency token by ID
    const currencyToken = await db_1.models.ecosystemToken.findOne({
        where: { id: currency, status: true },
    });
    if (!currencyToken) {
        throw (0, error_1.createError)(404, "Currency token not found or inactive");
    }
    // 2) Find the pair token by ID
    const pairToken = await db_1.models.ecosystemToken.findOne({
        where: { id: pair, status: true },
    });
    if (!pairToken) {
        throw (0, error_1.createError)(404, "Pair token not found or inactive");
    }
    // 2.1) Check if a market with the same currency and pair already exists.
    // Using currencyToken.currency (instead of .symbol) based on your token schema.
    const existingMarket = await db_1.models.futuresMarket.findOne({
        where: {
            currency: currencyToken.currency,
            pair: pairToken.currency,
        },
    });
    if (existingMarket) {
        throw (0, error_1.createError)(409, "Futures market with the given currency and pair already exists.");
    }
    // 3) Store the new market
    try {
        return await (0, query_1.storeRecord)({
            model: "futuresMarket",
            data: {
                currency: currencyToken.currency,
                pair: pairToken.currency,
                isTrending,
                isHot,
                metadata,
                status: true,
            },
        });
    }
    catch (error) {
        if (error.name === "SequelizeUniqueConstraintError") {
            throw (0, error_1.createError)(409, "Futures market already exists.");
        }
        throw error;
    }
};
