"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const constants_1 = require("@b/utils/constants");
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
// Safe import for ecosystem modules
let getFiltered;
try {
    const module = require("@b/api/(ext)/ecosystem/utils/scylla/query");
    getFiltered = module.getFiltered;
}
catch (e) {
    // Ecosystem extension not available
}
exports.metadata = {
    summary: "List all futures orders",
    operationId: "listFuturesOrders",
    tags: ["Admin", "Futures Orders"],
    parameters: constants_1.crudParameters,
    responses: {
        200: {
            description: "Futures orders retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            items: {
                                type: "array",
                                items: { type: "object", properties: utils_1.orderSchema },
                            },
                            pagination: {
                                type: "object",
                                properties: {
                                    totalItems: { type: "number" },
                                    currentPage: { type: "number" },
                                    perPage: { type: "number" },
                                    totalPages: { type: "number" },
                                },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Futures Orders"),
        500: query_1.serverErrorResponse,
    },
    permission: "view.futures.order",
    requiresAuth: true,
};
const keyspace = process.env.SCYLLA_FUTURES_KEYSPACE || "futures";
exports.default = async (data) => {
    const { query } = data;
    if (!getFiltered) {
        return {
            error: "Ecosystem extension not available",
            status: 500
        };
    }
    const table = "orders";
    const partitionKeys = ["userId"];
    const result = await getFiltered({
        table,
        query,
        filter: query.filter,
        sortField: query.sortField || "createdAt",
        sortOrder: query.sortOrder || "DESC",
        perPage: Number(query.perPage) || 10,
        allowFiltering: true,
        keyspace,
        partitionKeys,
        transformColumns: [
            "amount",
            "cost",
            "fee",
            "filled",
            "price",
            "remaining",
            "leverage",
            "stopLossPrice",
            "takeProfitPrice",
        ],
        nonStringLikeColumns: ["userId"],
    });
    return result;
};
