"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Get ICO Investment Limits",
    description: "Retrieves the current ICO investment limit settings",
    operationId: "getIcoLimits",
    tags: ["Admin", "ICO", "Settings"],
    requiresAuth: true,
    requiresAdmin: true,
    responses: {
        200: {
            description: "ICO limits retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            minInvestment: { type: "number" },
                            maxInvestment: { type: "number" },
                            maxPerUser: { type: "number" },
                            softCapPercentage: { type: "number" },
                            refundGracePeriod: { type: "number" },
                            vestingEnabled: { type: "boolean" },
                            defaultVestingMonths: { type: "number" },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        403: { description: "Forbidden" },
        500: { description: "Internal Server Error" },
    },
};
exports.default = async (data) => {
    var _a, _b;
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({
            statusCode: 401,
            message: "Authentication required"
        });
    }
    // Check admin role through user model
    const fullUser = await db_1.models.user.findByPk(user.id, {
        include: [{ model: db_1.models.role, as: "role" }]
    });
    if (!fullUser || (((_a = fullUser.role) === null || _a === void 0 ? void 0 : _a.name) !== 'admin' && ((_b = fullUser.role) === null || _b === void 0 ? void 0 : _b.name) !== 'super_admin')) {
        throw (0, error_1.createError)({
            statusCode: 403,
            message: "Admin privileges required"
        });
    }
    // Get all ICO-related settings
    const settingKeys = [
        'icoMinInvestment',
        'icoMaxInvestment',
        'icoMaxPerUser',
        'icoSoftCapPercentage',
        'icoRefundGracePeriod',
        'icoVestingEnabled',
        'icoDefaultVestingMonths',
    ];
    const settings = await db_1.models.settings.findAll({
        where: { key: settingKeys },
    });
    // Convert to object with defaults
    const settingsMap = settings.reduce((acc, setting) => {
        acc[setting.key] = setting.value;
        return acc;
    }, {});
    return {
        minInvestment: parseFloat(settingsMap.icoMinInvestment || '10'),
        maxInvestment: parseFloat(settingsMap.icoMaxInvestment || '100000'),
        maxPerUser: parseFloat(settingsMap.icoMaxPerUser || '50000'),
        softCapPercentage: parseFloat(settingsMap.icoSoftCapPercentage || '30'),
        refundGracePeriod: parseInt(settingsMap.icoRefundGracePeriod || '7'),
        vestingEnabled: settingsMap.icoVestingEnabled === 'true',
        defaultVestingMonths: parseInt(settingsMap.icoDefaultVestingMonths || '12'),
    };
};
