"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Update ICO Investment Limits",
    description: "Updates the ICO investment limit settings",
    operationId: "updateIcoLimits",
    tags: ["Admin", "ICO", "Settings"],
    requiresAuth: true,
    requiresAdmin: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        minInvestment: { type: "number", minimum: 0 },
                        maxInvestment: { type: "number", minimum: 0 },
                        maxPerUser: { type: "number", minimum: 0 },
                        softCapPercentage: { type: "number", minimum: 0, maximum: 100 },
                        refundGracePeriod: { type: "number", minimum: 0 },
                        vestingEnabled: { type: "boolean" },
                        defaultVestingMonths: { type: "number", minimum: 0 },
                    },
                },
            },
        },
    },
    responses: {
        200: {
            description: "ICO limits updated successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: { type: "string" },
                        },
                    },
                },
            },
        },
        400: { description: "Bad Request" },
        401: { description: "Unauthorized" },
        403: { description: "Forbidden" },
        500: { description: "Internal Server Error" },
    },
};
exports.default = async (data) => {
    var _a, _b;
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({
            statusCode: 401,
            message: "Authentication required"
        });
    }
    // Check admin role through user model
    const fullUser = await db_1.models.user.findByPk(user.id, {
        include: [{ model: db_1.models.role, as: "role" }]
    });
    if (!fullUser || (((_a = fullUser.role) === null || _a === void 0 ? void 0 : _a.name) !== 'admin' && ((_b = fullUser.role) === null || _b === void 0 ? void 0 : _b.name) !== 'super_admin')) {
        throw (0, error_1.createError)({
            statusCode: 403,
            message: "Admin privileges required"
        });
    }
    const { minInvestment, maxInvestment, maxPerUser, softCapPercentage, refundGracePeriod, vestingEnabled, defaultVestingMonths, } = body;
    // Validate limits
    if (minInvestment !== undefined && minInvestment < 0) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Minimum investment cannot be negative"
        });
    }
    if (maxInvestment !== undefined && minInvestment !== undefined && maxInvestment < minInvestment) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Maximum investment must be greater than minimum investment"
        });
    }
    if (softCapPercentage !== undefined && (softCapPercentage < 0 || softCapPercentage > 100)) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Soft cap percentage must be between 0 and 100"
        });
    }
    const transaction = await db_1.sequelize.transaction();
    try {
        // Update settings
        const updates = [
            { key: 'icoMinInvestment', value: minInvestment === null || minInvestment === void 0 ? void 0 : minInvestment.toString() },
            { key: 'icoMaxInvestment', value: maxInvestment === null || maxInvestment === void 0 ? void 0 : maxInvestment.toString() },
            { key: 'icoMaxPerUser', value: maxPerUser === null || maxPerUser === void 0 ? void 0 : maxPerUser.toString() },
            { key: 'icoSoftCapPercentage', value: softCapPercentage === null || softCapPercentage === void 0 ? void 0 : softCapPercentage.toString() },
            { key: 'icoRefundGracePeriod', value: refundGracePeriod === null || refundGracePeriod === void 0 ? void 0 : refundGracePeriod.toString() },
            { key: 'icoVestingEnabled', value: vestingEnabled === null || vestingEnabled === void 0 ? void 0 : vestingEnabled.toString() },
            { key: 'icoDefaultVestingMonths', value: defaultVestingMonths === null || defaultVestingMonths === void 0 ? void 0 : defaultVestingMonths.toString() },
        ].filter(update => update.value !== undefined);
        for (const update of updates) {
            await db_1.models.settings.upsert({
                key: update.key,
                value: update.value,
            }, { transaction });
        }
        // Create audit log
        await db_1.models.icoAdminActivity.create({
            type: "SETTINGS_UPDATED",
            offeringId: null,
            offeringName: "ICO Limits",
            adminId: user.id,
            details: JSON.stringify({
                updates: updates.reduce((acc, u) => {
                    acc[u.key] = u.value;
                    return acc;
                }, {}),
            }),
        }, { transaction });
        await transaction.commit();
        return {
            message: "ICO limits updated successfully",
        };
    }
    catch (err) {
        await transaction.rollback();
        throw err;
    }
};
