"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const sequelize_1 = require("sequelize");
exports.metadata = {
    summary: "Get External Pool Performance",
    description: "Retrieves external pool performance data with optional filtering by pool ID and date range.",
    operationId: "getExternalPoolPerformance",
    tags: ["Staking", "Admin", "Performance"],
    requiresAuth: true,
    parameters: [
        {
            index: 0,
            name: "poolId",
            in: "query",
            required: false,
            schema: { type: "string" },
            description: "Filter performance by pool ID",
        },
        {
            index: 1,
            name: "startDate",
            in: "query",
            required: false,
            schema: { type: "string", format: "date" },
            description: "Filter performance by start date",
        },
        {
            index: 2,
            name: "endDate",
            in: "query",
            required: false,
            schema: { type: "string", format: "date" },
            description: "Filter performance by end date",
        },
    ],
    responses: {
        200: {
            description: "Performance data retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            items: {
                                type: "array",
                                items: { type: "object" },
                            },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
    permission: "view.staking.performance",
};
exports.default = async (data) => {
    const { user, query } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    try {
        // Build filter conditions
        const where = {};
        if (query === null || query === void 0 ? void 0 : query.poolId) {
            where.poolId = query.poolId;
        }
        if (query === null || query === void 0 ? void 0 : query.startDate) {
            where.date = {
                ...where.date,
                [sequelize_1.Op.gte]: new Date(query.startDate),
            };
        }
        if (query === null || query === void 0 ? void 0 : query.endDate) {
            where.date = {
                ...where.date,
                [sequelize_1.Op.lte]: new Date(query.endDate),
            };
        }
        // Fetch external performance data with their pool
        const performances = await db_1.models.stakingExternalPoolPerformance.findAll({
            where,
            include: [
                {
                    model: db_1.models.stakingPool,
                    as: "pool",
                },
            ],
            order: [["date", "DESC"]],
        });
        return performances;
    }
    catch (error) {
        console.error("Error fetching external pool performance:", error);
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Failed to fetch external pool performance",
        });
    }
};
