"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const notifications_1 = require("@b/utils/notifications");
exports.metadata = {
    summary: "Add External Pool Performance",
    description: "Creates a new external pool performance record.",
    operationId: "addExternalPoolPerformance",
    tags: ["Staking", "Admin", "Performance"],
    requiresAuth: true,
    requestBody: {
        description: "External pool performance data",
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        poolId: { type: "string" },
                        date: { type: "string", format: "date-time" },
                        apr: { type: "number" },
                        totalStaked: { type: "number" },
                        profit: { type: "number" },
                        notes: { type: "string" },
                    },
                    required: ["poolId", "date", "apr", "totalStaked", "profit"],
                },
            },
        },
    },
    responses: {
        201: {
            description: "Performance record created successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                    },
                },
            },
        },
        400: { description: "Bad Request" },
        401: { description: "Unauthorized" },
        404: { description: "Pool not found" },
        500: { description: "Internal Server Error" },
    },
    permission: "create.staking.performance",
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    if (!body) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Request body is required" });
    }
    const { poolId, date, apr, totalStaked, profit, notes = "" } = body;
    if (!poolId ||
        !date ||
        apr === undefined ||
        totalStaked === undefined ||
        profit === undefined) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "poolId, date, apr, totalStaked, and profit are required",
        });
    }
    try {
        // Check if the pool exists
        const pool = await db_1.models.stakingPool.findByPk(poolId);
        if (!pool) {
            throw (0, error_1.createError)({ statusCode: 404, message: "Pool not found" });
        }
        // Create the performance record
        const performance = await db_1.models.stakingExternalPoolPerformance.create({
            poolId,
            date,
            apr,
            totalStaked,
            profit,
            notes,
        });
        // Fetch the created record with its pool
        const createdPerformance = await db_1.models.stakingExternalPoolPerformance.findOne({
            where: { id: performance.id },
            include: [
                {
                    model: db_1.models.stakingPool,
                    as: "pool",
                },
            ],
        });
        // Create a notification for the admin
        try {
            await (0, notifications_1.createNotification)({
                userId: user.id,
                relatedId: performance.id,
                type: "system",
                title: "Pool Performance Added",
                message: `New performance record added for ${pool.name} with ${apr}% APR.`,
                details: "The performance record has been created successfully.",
                link: `/admin/staking/performance`,
                actions: [
                    {
                        label: "View Performance",
                        link: `/admin/staking/performance`,
                        primary: true,
                    },
                ],
            });
        }
        catch (notifErr) {
            console.error("Failed to create notification for performance record", notifErr);
            // Continue execution even if notification fails
        }
        return createdPerformance;
    }
    catch (error) {
        if (error.statusCode === 404) {
            throw error;
        }
        console.error("Error creating external pool performance:", error);
        throw (0, error_1.createError)({
            statusCode: 500,
            message: error.message,
        });
    }
};
