"use strict";
// DepositUtils.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.processTransaction = processTransaction;
exports.createTransactionDetails = createTransactionDetails;
const ethers_1 = require("ethers");
const deposit_1 = require("@b/api/(ext)/ecosystem/utils/redis/deposit");
const blockchain_1 = require("@b/api/(ext)/ecosystem/utils/blockchain");
/**
 * Decodes and validates a transaction, ensures `to` matches our target address.
 * Enhanced with better error handling and validation.
 * Returns true if processing was successful, false otherwise.
 */
async function processTransaction(contractType, txHash, provider, address, chain, decimals, feeDecimals, walletId) {
    // Input validation
    if (!txHash || !provider || !address || !chain || !walletId) {
        console.error(`[ERROR] Invalid parameters for processTransaction: txHash=${txHash}, address=${address}, chain=${chain}, walletId=${walletId}`);
        return false;
    }
    try {
        console.log(`[INFO] Processing ${contractType} transaction ${txHash} on ${chain}`);
        const tx = await provider.getTransaction(txHash);
        if (!tx) {
            console.error(`[ERROR] Transaction ${txHash} not found on ${chain}`);
            return false;
        }
        if (!tx.data) {
            console.error(`[ERROR] Transaction ${txHash} has no data field`);
            return false;
        }
        const decodedData = (0, blockchain_1.decodeTransactionData)(tx.data);
        const realTo = decodedData.to || tx.to;
        const amount = decodedData.amount || tx.value;
        if (!realTo || !address) {
            console.error(`[ERROR] Invalid transaction data for ${txHash}: realTo=${realTo}, address=${address}`);
            return false;
        }
        // Validate address match (case-insensitive)
        if (realTo.toLowerCase() !== address.toLowerCase()) {
            console.warn(`[WARN] Address mismatch for ${txHash}: expected=${address}, actual=${realTo}`);
            return false;
        }
        // Validate amount
        if (!amount || amount.toString() === "0") {
            console.warn(`[WARN] Zero or invalid amount for transaction ${txHash}`);
            return false;
        }
        const txDetails = await createTransactionDetails(contractType, walletId, tx, realTo, chain, decimals, feeDecimals, "DEPOSIT", amount);
        await (0, deposit_1.storeAndBroadcastTransaction)(txDetails, txHash);
        console.log(`[SUCCESS] Transaction ${txHash} processed successfully on ${chain}`);
        return true;
    }
    catch (error) {
        console.error(`[ERROR] Error processing transaction ${txHash} on ${chain}: ${error.message}`);
        return false;
    }
}
/**
 * Creates standardized transaction details with enhanced validation and error handling
 */
async function createTransactionDetails(contractType, walletId, tx, toAddress, chain, decimals, feeDecimals, type, amount = tx.amount) {
    var _a;
    try {
        // Input validation
        if (!contractType || !walletId || !tx || !toAddress || !chain || !type) {
            throw new Error("Missing required parameters for createTransactionDetails");
        }
        // Validate decimals
        if (decimals < 0 || decimals > 18) {
            console.warn(`[WARN] Unusual decimals value: ${decimals} for chain ${chain}`);
        }
        if (feeDecimals < 0 || feeDecimals > 18) {
            console.warn(`[WARN] Unusual fee decimals value: ${feeDecimals} for chain ${chain}`);
        }
        // Safe amount formatting with validation
        let formattedAmount = "0";
        try {
            if (amount && amount.toString() !== "0") {
                formattedAmount = ethers_1.ethers.formatUnits(amount.toString(), decimals);
                // Validate formatted amount
                if (isNaN(parseFloat(formattedAmount)) ||
                    parseFloat(formattedAmount) <= 0) {
                    console.warn(`[WARN] Invalid formatted amount ${formattedAmount} for transaction ${tx.hash}`);
                    formattedAmount = "0";
                }
            }
        }
        catch (error) {
            console.error(`[ERROR] Error formatting amount for transaction ${tx.hash}: ${error.message}`);
            formattedAmount = "0";
        }
        // Safe gas limit formatting
        let formattedGasLimit = "N/A";
        try {
            if (tx.gasLimit) {
                formattedGasLimit = tx.gasLimit.toString();
            }
        }
        catch (error) {
            console.warn(`[WARN] Error formatting gas limit for transaction ${tx.hash}: ${error.message}`);
        }
        // Safe gas price formatting
        let formattedGasPrice = "N/A";
        try {
            if (tx.gasPrice) {
                formattedGasPrice = ethers_1.ethers.formatUnits(tx.gasPrice.toString(), feeDecimals);
                // Validate gas price
                if (isNaN(parseFloat(formattedGasPrice)) ||
                    parseFloat(formattedGasPrice) < 0) {
                    console.warn(`[WARN] Invalid gas price ${formattedGasPrice} for transaction ${tx.hash}`);
                    formattedGasPrice = "N/A";
                }
            }
        }
        catch (error) {
            console.warn(`[WARN] Error formatting gas price for transaction ${tx.hash}: ${error.message}`);
        }
        const txDetails = {
            contractType,
            id: walletId,
            chain,
            hash: tx.hash,
            type,
            from: tx.from || "unknown",
            to: toAddress,
            amount: formattedAmount,
            gasLimit: formattedGasLimit,
            gasPrice: formattedGasPrice,
            timestamp: Math.floor(Date.now() / 1000),
            blockNumber: ((_a = tx.blockNumber) === null || _a === void 0 ? void 0 : _a.toString()) || "0",
            status: "PENDING", // Will be updated by verification process
        };
        console.log(`[INFO] Created transaction details for ${tx.hash}: amount=${formattedAmount}, chain=${chain}`);
        return txDetails;
    }
    catch (error) {
        console.error(`[ERROR] Error creating transaction details: ${error.message}`);
        throw error;
    }
}
