"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const utils_1 = require("@b/api/finance/wallet/utils");
const blockchain_1 = require("@b/api/(ext)/ecosystem/utils/blockchain");
const matchingEngine_1 = require("@b/api/(ext)/ecosystem/utils/matchingEngine");
const queries_1 = require("@b/api/(ext)/ecosystem/utils/scylla/queries");
const wallet_1 = require("@b/api/(ext)/ecosystem/utils/wallet");
const error_1 = require("@b/utils/error");
const query_1 = require("@b/utils/query");
exports.metadata = {
    summary: "Cancels all open trading orders",
    description: "Cancels all open trading orders for the user and refunds the unfulfilled amounts.",
    operationId: "cancelAllOrders",
    tags: ["Trading", "Orders"],
    responses: {
        200: {
            description: "All orders cancelled successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: { type: "string", description: "Success message" },
                            cancelledCount: { type: "number", description: "Number of orders cancelled" },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    try {
        // Get all orders for the user
        const allOrders = await (0, queries_1.getOrdersByUserId)(user.id);
        // Filter only OPEN orders
        const openOrders = allOrders.filter(order => order.status === "OPEN");
        if (openOrders.length === 0) {
            return {
                message: "No open orders to cancel",
                cancelledCount: 0,
            };
        }
        const matchingEngine = await matchingEngine_1.MatchingEngine.getInstance();
        let cancelledCount = 0;
        // Cancel each open order
        for (const order of openOrders) {
            try {
                const totalAmount = BigInt(order.amount);
                const remaining = BigInt(order.remaining);
                const totalCost = BigInt(order.cost);
                const side = order.side;
                const symbol = order.symbol;
                if (remaining === BigInt(0)) {
                    continue; // Skip fully filled orders
                }
                const [currency, pair] = symbol.split("/");
                let refundAmount = 0;
                if (side === "BUY") {
                    const leftoverRatio = Number(remaining) / Number(totalAmount);
                    refundAmount = (0, blockchain_1.fromBigInt)(totalCost) * leftoverRatio;
                }
                else {
                    const leftoverRatio = Number(remaining) / Number(totalAmount);
                    refundAmount = (0, blockchain_1.fromBigInt)(totalAmount) * leftoverRatio;
                }
                const refundCurrency = side === "BUY" ? pair : currency;
                const wallet = await (0, utils_1.getWallet)(user.id, "ECO", refundCurrency);
                if (!wallet) {
                    console.error(`Wallet not found for ${refundCurrency}, skipping order ${order.id}`);
                    continue;
                }
                // Cancel the order
                await (0, queries_1.cancelOrderByUuid)(user.id, order.id, typeof order.createdAt === 'string' ? order.createdAt : order.createdAt.toISOString(), symbol, BigInt(order.price), side, totalAmount);
                // Refund the leftover funds
                await (0, wallet_1.updateWalletBalance)(wallet, refundAmount, "add");
                // Remove from orderbook and internal queues
                await matchingEngine.handleOrderCancellation(order.id, symbol);
                cancelledCount++;
            }
            catch (error) {
                console.error(`Failed to cancel order ${order.id}:`, error.message);
                // Continue with other orders even if one fails
            }
        }
        return {
            message: `Successfully cancelled ${cancelledCount} order(s)`,
            cancelledCount,
        };
    }
    catch (error) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: `Failed to cancel orders: ${error.message}`,
        });
    }
};
