"use strict";
/**
 * Mempool.space Provider for UTXO chains
 * Free, unlimited API with excellent support for Bitcoin and Litecoin
 * https://mempool.space/docs/api
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MempoolProvider = void 0;
const logger_1 = require("@b/utils/logger");
class MempoolProvider {
    constructor(chain) {
        this.timeout = 30000;
        this.chain = chain;
        this.baseURL = this.getBaseURL(chain);
    }
    getBaseURL(chain) {
        const urls = {
            'BTC': process.env.BTC_NETWORK === 'testnet'
                ? 'https://mempool.space/testnet/api'
                : 'https://mempool.space/api',
            'LTC': 'https://litecoinspace.org/api',
        };
        if (!urls[chain]) {
            throw new Error(`Mempool provider not available for ${chain}`);
        }
        return urls[chain];
    }
    getName() {
        return `Mempool.space (${this.chain})`;
    }
    async fetchFromAPI(endpoint, options = {}) {
        const url = `${this.baseURL}${endpoint}`;
        try {
            const response = await fetch(url, {
                ...options,
                signal: AbortSignal.timeout(this.timeout),
            });
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            const contentType = response.headers.get('content-type');
            if (contentType === null || contentType === void 0 ? void 0 : contentType.includes('application/json')) {
                return await response.json();
            }
            else {
                return await response.text();
            }
        }
        catch (error) {
            (0, logger_1.logError)('mempool_provider_fetch', error, __filename);
            throw error;
        }
    }
    async fetchTransactions(address) {
        try {
            const txs = await this.fetchFromAPI(`/address/${address}/txs`);
            const currentHeight = await this.getCurrentBlockHeight();
            return txs.map((tx) => {
                const confirmations = tx.status.confirmed
                    ? currentHeight - tx.status.block_height + 1
                    : 0;
                // Calculate value for this address
                let value = 0;
                tx.vout.forEach((output) => {
                    if (output.scriptpubkey_address === address) {
                        value += output.value;
                    }
                });
                return {
                    hash: tx.txid,
                    blockHeight: tx.status.block_height,
                    value: value, // in satoshis
                    confirmedTime: tx.status.block_time ? new Date(tx.status.block_time * 1000).toISOString() : undefined,
                    spent: false, // Would need additional check
                    confirmations: confirmations,
                    fee: tx.fee,
                };
            });
        }
        catch (error) {
            (0, logger_1.logError)('mempool_fetch_transactions', error, __filename);
            return [];
        }
    }
    async fetchTransaction(txHash) {
        try {
            const tx = await this.fetchFromAPI(`/tx/${txHash}`);
            const currentHeight = await this.getCurrentBlockHeight();
            const confirmations = tx.status.confirmed
                ? currentHeight - tx.status.block_height + 1
                : 0;
            // Parse inputs
            const inputs = tx.vin.map((input) => {
                var _a, _b, _c;
                return ({
                    prev_hash: input.txid,
                    prevHash: input.txid,
                    output_index: input.vout,
                    outputIndex: input.vout,
                    output_value: ((_a = input.prevout) === null || _a === void 0 ? void 0 : _a.value) || 0, // Already in satoshis
                    addresses: ((_b = input.prevout) === null || _b === void 0 ? void 0 : _b.scriptpubkey_address) ? [input.prevout.scriptpubkey_address] : [],
                    script: (_c = input.prevout) === null || _c === void 0 ? void 0 : _c.scriptpubkey,
                });
            });
            // Parse outputs
            const outputs = tx.vout.map((output) => ({
                value: output.value, // Already in satoshis
                addresses: output.scriptpubkey_address ? [output.scriptpubkey_address] : [],
                script: output.scriptpubkey,
                spent: output.spent || false,
                spent_by: output.spent_by,
                spender: output.spent_by,
            }));
            return {
                hash: tx.txid,
                block_height: tx.status.block_height,
                confirmations: confirmations,
                fee: tx.fee, // Already in satoshis
                inputs: inputs,
                outputs: outputs,
            };
        }
        catch (error) {
            (0, logger_1.logError)('mempool_fetch_transaction', error, __filename);
            return null;
        }
    }
    async fetchRawTransaction(txHash) {
        try {
            const hex = await this.fetchFromAPI(`/tx/${txHash}/hex`);
            return hex;
        }
        catch (error) {
            (0, logger_1.logError)('mempool_fetch_raw_transaction', error, __filename);
            throw error;
        }
    }
    async getBalance(address) {
        try {
            const data = await this.fetchFromAPI(`/address/${address}`);
            const funded = data.chain_stats.funded_txo_sum || 0;
            const spent = data.chain_stats.spent_txo_sum || 0;
            return funded - spent; // Returns satoshis
        }
        catch (error) {
            (0, logger_1.logError)('mempool_get_balance', error, __filename);
            return 0;
        }
    }
    async getUTXOs(address) {
        try {
            const utxos = await this.fetchFromAPI(`/address/${address}/utxo`);
            const currentHeight = await this.getCurrentBlockHeight();
            return utxos.map((utxo) => ({
                txid: utxo.txid,
                vout: utxo.vout,
                value: utxo.value, // in satoshis
                confirmations: utxo.status.confirmed
                    ? currentHeight - utxo.status.block_height + 1
                    : 0,
                script: utxo.scriptpubkey,
            }));
        }
        catch (error) {
            (0, logger_1.logError)('mempool_get_utxos', error, __filename);
            return [];
        }
    }
    async broadcastTransaction(rawTxHex) {
        try {
            const txid = await this.fetchFromAPI('/tx', {
                method: 'POST',
                headers: {
                    'Content-Type': 'text/plain',
                },
                body: rawTxHex,
            });
            return {
                success: true,
                txid: txid,
            };
        }
        catch (error) {
            (0, logger_1.logError)('mempool_broadcast_transaction', error, __filename);
            return {
                success: false,
                txid: null,
                error: error.message,
            };
        }
    }
    async getFeeRate() {
        try {
            const fees = await this.fetchFromAPI('/v1/fees/recommended');
            // Return the "half hour" fee rate as default
            // You can choose: fastestFee, halfHourFee, hourFee, economyFee, minimumFee
            const feeRatePriority = process.env.BTC_FEE_RATE_PRIORITY || 'halfHourFee';
            return fees[feeRatePriority] || fees.halfHourFee || fees.fastestFee;
        }
        catch (error) {
            (0, logger_1.logError)('mempool_get_fee_rate', error, __filename);
            return 1; // Default 1 sat/vByte
        }
    }
    async getCurrentBlockHeight() {
        try {
            const height = await this.fetchFromAPI('/blocks/tip/height');
            return parseInt(height);
        }
        catch (error) {
            (0, logger_1.logError)('mempool_get_block_height', error, __filename);
            return 0;
        }
    }
    async isAvailable() {
        try {
            await this.fetchFromAPI('/blocks/tip/height');
            return true;
        }
        catch (error) {
            return false;
        }
    }
}
exports.MempoolProvider = MempoolProvider;
