"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.calculateTransactionFees = calculateTransactionFees;
exports.validateAccountOwnership = validateAccountOwnership;
exports.getUserWallet = getUserWallet;
exports.createForexTransaction = createForexTransaction;
const exchange_1 = __importDefault(require("@b/utils/exchange"));
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const cron_1 = require("@b/utils/cron");
/**
 * Calculate transaction fees for forex operations
 * Shared logic between deposit and withdrawal
 */
async function calculateTransactionFees(type, currency, chain, amount, transaction) {
    var _a, _b, _c;
    let currencyData;
    let taxAmount = 0;
    let precision = 8;
    switch (type) {
        case "FIAT":
            currencyData = await db_1.models.currency.findOne({
                where: { id: currency },
                transaction,
            });
            if (!currencyData || !currencyData.price) {
                await (0, cron_1.fetchFiatCurrencyPrices)();
                currencyData = await db_1.models.currency.findOne({
                    where: { id: currency },
                    transaction,
                });
                if (!currencyData || !currencyData.price)
                    throw new Error("Currency processing failed");
            }
            precision = 2;
            break;
        case "SPOT":
            currencyData = await db_1.models.exchangeCurrency.findOne({
                where: { currency: currency },
                transaction,
            });
            if (!currencyData || !currencyData.price) {
                await (0, cron_1.processCurrenciesPrices)();
                currencyData = await db_1.models.exchangeCurrency.findOne({
                    where: { currency: currency },
                    transaction,
                });
                if (!currencyData || !currencyData.price)
                    throw new Error("Currency processing failed");
            }
            const exchange = await exchange_1.default.startExchange();
            const provider = await exchange_1.default.getProvider();
            if (!exchange)
                throw (0, error_1.createError)(500, "Exchange not found");
            const currencies = await exchange.fetchCurrencies();
            const isXt = provider === "xt";
            const exchangeCurrency = Object.values(currencies).find((c) => isXt ? c.code === currency : c.id === currency);
            if (!exchangeCurrency)
                throw (0, error_1.createError)(404, "Currency not found");
            let fixedFee = 0;
            switch (provider) {
                case "binance":
                case "kucoin":
                    if (chain && exchangeCurrency.networks) {
                        fixedFee =
                            ((_a = exchangeCurrency.networks[chain]) === null || _a === void 0 ? void 0 : _a.fee) ||
                                ((_c = (_b = exchangeCurrency.networks[chain]) === null || _b === void 0 ? void 0 : _b.fees) === null || _c === void 0 ? void 0 : _c.withdraw) ||
                                0;
                    }
                    break;
                default:
                    break;
            }
            const parsedAmount = parseFloat(amount.toString());
            const percentageFee = currencyData.fee || 0;
            taxAmount = parseFloat(Math.max((parsedAmount * percentageFee) / 100 + fixedFee, 0).toFixed(2));
            precision = currencyData.precision || 8;
            break;
        default:
            throw new Error("Invalid wallet type");
    }
    const total = amount + taxAmount;
    return {
        currencyData,
        taxAmount,
        total,
        precision,
    };
}
/**
 * Validate forex account ownership
 */
async function validateAccountOwnership(accountId, userId, transaction) {
    const account = await db_1.models.forexAccount.findByPk(accountId, {
        transaction,
    });
    if (!account) {
        throw new Error("Account not found");
    }
    if (account.userId !== userId) {
        throw (0, error_1.createError)({
            statusCode: 403,
            message: "Access denied: You can only access your own forex accounts"
        });
    }
    return account;
}
/**
 * Get or create user wallet
 */
async function getUserWallet(userId, type, currency, transaction) {
    const wallet = await db_1.models.wallet.findOne({
        where: { userId, type, currency },
        transaction,
    });
    if (!wallet) {
        throw new Error("Wallet not found");
    }
    return wallet;
}
/**
 * Create forex transaction record
 */
async function createForexTransaction(userId, walletId, type, amount, fee, accountId, metadata, transaction) {
    const description = type === "FOREX_DEPOSIT"
        ? `Deposit to Forex account ${accountId}`
        : `Withdraw from Forex account ${accountId}`;
    return await db_1.models.transaction.create({
        userId,
        walletId,
        type,
        status: "PENDING",
        amount,
        fee,
        description,
        metadata: JSON.stringify(metadata),
    }, { transaction });
}
