"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const Websocket_1 = require("@b/handler/Websocket");
const matchingEngine_1 = require("@b/api/(ext)/futures/utils/matchingEngine");
const orderbook_1 = require("@b/api/(ext)/futures/utils/queries/orderbook");
const db_1 = require("@b/db");
exports.metadata = {};
class UnifiedFuturesMarketDataHandler {
    constructor() {
        this.activeSubscriptions = new Map(); // symbol -> Set<dataTypes>
        this.intervalMap = new Map(); // symbol -> interval
        this.engine = null;
    }
    static getInstance() {
        if (!UnifiedFuturesMarketDataHandler.instance) {
            UnifiedFuturesMarketDataHandler.instance = new UnifiedFuturesMarketDataHandler();
        }
        return UnifiedFuturesMarketDataHandler.instance;
    }
    async initializeEngine() {
        if (!this.engine) {
            this.engine = await matchingEngine_1.FuturesMatchingEngine.getInstance();
        }
    }
    async fetchAndBroadcastData(symbol, dataTypes) {
        try {
            await this.initializeEngine();
            const fetchPromises = Array.from(dataTypes).map(async (type) => {
                try {
                    switch (type) {
                        case "orderbook":
                            const orderbook = await (0, orderbook_1.getOrderBook)(symbol);
                            Websocket_1.messageBroker.broadcastToSubscribedClients(`/api/futures/market`, { type: "orderbook", symbol }, {
                                stream: "orderbook",
                                data: orderbook,
                            });
                            break;
                        case "trades":
                            Websocket_1.messageBroker.broadcastToSubscribedClients(`/api/futures/market`, { type: "trades", symbol }, {
                                stream: "trades",
                                data: [],
                            });
                            break;
                        case "ticker":
                            const ticker = await this.engine.getTicker(symbol);
                            Websocket_1.messageBroker.broadcastToSubscribedClients(`/api/futures/market`, { type: "ticker", symbol }, {
                                stream: "ticker",
                                data: ticker,
                            });
                            break;
                    }
                }
                catch (error) {
                    console.error(`Error fetching ${type} data for ${symbol}:`, error);
                }
            });
            await Promise.allSettled(fetchPromises);
        }
        catch (error) {
            console.error(`Error in fetchAndBroadcastData for ${symbol}:`, error);
        }
    }
    startDataFetching(symbol) {
        // Clear existing interval if any
        if (this.intervalMap.has(symbol)) {
            clearInterval(this.intervalMap.get(symbol));
        }
        // Start new interval for this symbol
        const interval = setInterval(async () => {
            const dataTypes = this.activeSubscriptions.get(symbol);
            if (dataTypes && dataTypes.size > 0) {
                await this.fetchAndBroadcastData(symbol, dataTypes);
            }
        }, 500); // Fetch every 500ms
        this.intervalMap.set(symbol, interval);
    }
    async addSubscription(symbol, type) {
        // Validate that the symbol exists in the database and is enabled
        if (!symbol) {
            console.warn("No symbol provided in futures subscription request");
            return;
        }
        const [currency, pair] = symbol.split("/");
        if (!currency || !pair) {
            console.warn(`Invalid symbol format: ${symbol}. Expected format: CURRENCY/PAIR`);
            return;
        }
        const market = await db_1.models.futuresMarket.findOne({
            where: {
                currency,
                pair,
                status: true // Only allow enabled markets
            }
        });
        if (!market) {
            console.warn(`Futures market ${symbol} not found in database or is disabled. Skipping subscription.`);
            return;
        }
        // Add this data type to the symbol's subscription set
        if (!this.activeSubscriptions.has(symbol)) {
            this.activeSubscriptions.set(symbol, new Set([type]));
            // Start data fetching for this symbol
            this.startDataFetching(symbol);
        }
        else {
            // Add the data type to the existing symbol's subscription set
            this.activeSubscriptions.get(symbol).add(type);
        }
        console.log(`Added ${type} subscription for ${symbol}. Active types:`, Array.from(this.activeSubscriptions.get(symbol)));
        // Immediately fetch data for the new subscription
        await this.fetchAndBroadcastData(symbol, new Set([type]));
    }
    removeSubscription(symbol, type) {
        if (this.activeSubscriptions.has(symbol)) {
            this.activeSubscriptions.get(symbol).delete(type);
            // If no more data types for this symbol, remove the symbol entirely
            if (this.activeSubscriptions.get(symbol).size === 0) {
                this.activeSubscriptions.delete(symbol);
                // Clear the interval
                if (this.intervalMap.has(symbol)) {
                    clearInterval(this.intervalMap.get(symbol));
                    this.intervalMap.delete(symbol);
                }
                console.log(`Removed all subscriptions for ${symbol}`);
            }
            else {
                console.log(`Removed ${type} subscription for ${symbol}. Remaining types:`, Array.from(this.activeSubscriptions.get(symbol)));
            }
        }
    }
    stop() {
        // Clear all intervals
        this.intervalMap.forEach((interval) => clearInterval(interval));
        this.intervalMap.clear();
        this.activeSubscriptions.clear();
    }
}
exports.default = async (data, message) => {
    if (typeof message === "string") {
        message = JSON.parse(message);
    }
    const { type, symbol } = message.payload;
    if (!type || !symbol) {
        console.error("Invalid message structure: type or symbol is missing");
        return;
    }
    const handler = UnifiedFuturesMarketDataHandler.getInstance();
    await handler.addSubscription(symbol, type);
};
