"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const sequelize_1 = require("sequelize");
exports.metadata = {
    summary: "Get Creator ICO Offerings",
    description: "Retrieves ICO offerings for the authenticated creator, grouped by status (active, pending, completed) along with currentRaised for each offering.",
    operationId: "getCreatorOfferings",
    tags: ["ICO", "Creator", "Offerings"],
    requiresAuth: true,
    responses: {
        200: {
            description: "Creator offerings retrieved successfully.",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            active: { type: "array", items: { type: "object" } },
                            pending: { type: "array", items: { type: "object" } },
                            completed: { type: "array", items: { type: "object" } },
                        },
                    },
                },
            },
        },
        401: { description: "Unauthorized" },
        500: { description: "Internal Server Error" },
    },
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    // Fetch offerings for the current creator.
    const offerings = await db_1.models.icoTokenOffering.findAll({
        where: { userId: user.id },
        raw: true,
    });
    // If there are no offerings, return empty arrays.
    if (!offerings.length) {
        return { active: [], pending: [], completed: [] };
    }
    // Get a list of offering IDs.
    const offeringIds = offerings.map((o) => o.id);
    // Query transactions to calculate currentRaised for each offering (excluding "REJECTED" transactions).
    const currentRaisedData = await db_1.models.icoTransaction.findAll({
        attributes: [
            "offeringId",
            [(0, sequelize_1.fn)("SUM", (0, sequelize_1.literal)("price * amount")), "currentRaised"],
        ],
        where: {
            offeringId: { [sequelize_1.Op.in]: offeringIds },
            status: { [sequelize_1.Op.not]: ["REJECTED"] },
        },
        group: ["offeringId"],
        raw: true,
    });
    // Create a mapping from offeringId to currentRaised.
    const raisedMap = {};
    currentRaisedData.forEach((row) => {
        raisedMap[row.offeringId] = parseFloat(row.currentRaised) || 0;
    });
    // Attach currentRaised to each offering.
    const offeringsWithRaised = offerings.map((offering) => ({
        ...offering,
        currentRaised: raisedMap[offering.id] || 0,
    }));
    // Group offerings by status.
    const active = offeringsWithRaised.filter((o) => o.status === "ACTIVE");
    const pending = offeringsWithRaised.filter((o) => o.status === "PENDING");
    const completed = offeringsWithRaised.filter((o) => o.status === "SUCCESS");
    return { active, pending, completed };
};
