"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const sequelize_1 = require("sequelize");
const query_1 = require("@b/utils/query");
exports.metadata = {
    summary: "Get P2P Dashboard Stats",
    description: "Retrieves various trade counts and stats for the authenticated user.",
    operationId: "getP2PDashboardStats",
    tags: ["P2P", "Dashboard"],
    responses: {
        200: { description: "Dashboard stats retrieved successfully." },
        401: query_1.unauthorizedResponse,
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        return {
            statusCode: 401,
            message: "Unauthorized",
        };
    }
    try {
        // Check if P2P models exist
        if (!db_1.models.p2pTrade) {
            return {
                statusCode: 500,
                message: "P2P extension is not properly installed or configured",
            };
        }
        let totalTrades = 0;
        let activeTrades = 0;
        let completedTrades = 0;
        let wallets = [];
        try {
            totalTrades = await db_1.models.p2pTrade.count({
                where: {
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
            });
        }
        catch (error) {
            console.error("Error fetching total trades:", error);
        }
        try {
            activeTrades = await db_1.models.p2pTrade.count({
                where: {
                    status: "PENDING",
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
            });
        }
        catch (error) {
            console.error("Error fetching active trades:", error);
        }
        try {
            completedTrades = await db_1.models.p2pTrade.count({
                where: {
                    status: "COMPLETED",
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
            });
        }
        catch (error) {
            console.error("Error fetching completed trades:", error);
        }
        // Fetch user wallets for P2P trading
        try {
            wallets = await db_1.models.wallet.findAll({
                where: {
                    userId: user.id,
                    type: { [sequelize_1.Op.in]: ["FIAT", "SPOT", "ECO"] },
                },
                attributes: [
                    "id",
                    "type",
                    "currency",
                    "balance",
                    "inOrder",
                    "status",
                ],
                raw: true,
            });
        }
        catch (error) {
            console.error("Error fetching user wallets:", error);
            wallets = [];
        }
        return {
            totalTrades,
            activeTrades,
            completedTrades,
            wallets: wallets.map((wallet) => ({
                id: wallet.id,
                type: wallet.type,
                currency: wallet.currency,
                balance: parseFloat(wallet.balance || 0),
                inOrder: parseFloat(wallet.inOrder || 0),
                availableBalance: parseFloat(wallet.balance || 0) - parseFloat(wallet.inOrder || 0),
                status: wallet.status,
            })),
        };
    }
    catch (err) {
        console.error("P2P Dashboard Stats API Error:", err);
        return {
            statusCode: 500,
            message: "Internal Server Error: " + err.message,
        };
    }
};
