"use strict";
// backend/src/api/p2p/offers/create.post.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
exports.default = handler;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const utils_1 = require("@b/api/finance/wallet/utils");
const sequelize_1 = require("sequelize");
const cache_1 = require("@b/utils/cache");
exports.metadata = {
    summary: "Create a P2P Offer",
    description: "Creates a new offer with structured configurations for the authenticated user, and associates payment methods.",
    operationId: "createP2POffer",
    tags: ["P2P", "Offer"],
    requiresAuth: true,
    middleware: ["p2pOfferCreateRateLimit"],
    requestBody: {
        description: "Complete P2P offer payload",
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        type: { type: "string", enum: ["BUY", "SELL"] },
                        currency: { type: "string" },
                        walletType: { type: "string", enum: ["FIAT", "SPOT", "ECO"] },
                        amountConfig: {
                            type: "object",
                            properties: {
                                total: { type: "number" },
                                min: { type: "number" },
                                max: { type: "number" },
                                availableBalance: { type: "number" },
                            },
                            required: ["total"],
                        },
                        priceConfig: {
                            type: "object",
                            properties: {
                                model: { type: "string", enum: ["FIXED", "MARGIN"] },
                                value: { type: "number" },
                                marketPrice: { type: "number" },
                                finalPrice: { type: "number" },
                            },
                            required: ["model", "value", "finalPrice"],
                        },
                        tradeSettings: {
                            type: "object",
                            properties: {
                                autoCancel: { type: "number" },
                                kycRequired: { type: "boolean" },
                                visibility: {
                                    type: "string",
                                    enum: ["PUBLIC", "PRIVATE"],
                                },
                                termsOfTrade: { type: "string", minLength: 1 },
                                additionalNotes: { type: "string" },
                            },
                            required: ["autoCancel", "kycRequired", "visibility", "termsOfTrade"],
                        },
                        locationSettings: {
                            type: "object",
                            properties: {
                                country: { type: "string", minLength: 1 },
                                region: { type: "string" },
                                city: { type: "string" },
                                restrictions: { type: "array", items: { type: "string" } },
                            },
                            required: ["country"],
                        },
                        userRequirements: {
                            type: "object",
                            properties: {
                                minCompletedTrades: { type: "number" },
                                minSuccessRate: { type: "number" },
                                minAccountAge: { type: "number" },
                                trustedOnly: { type: "boolean" },
                            },
                        },
                        paymentMethodIds: {
                            type: "array",
                            items: { type: "string", format: "uuid" },
                            description: "Array of P2P payment‐method IDs to attach",
                            minItems: 1,
                        },
                    },
                    required: [
                        "type",
                        "currency",
                        "walletType",
                        "amountConfig",
                        "priceConfig",
                        "tradeSettings",
                        "locationSettings",
                        "paymentMethodIds",
                    ],
                },
            },
        },
    },
    responses: {
        200: { description: "Offer created successfully." },
        400: { description: "Bad Request." },
        401: { description: "Unauthorized." },
        500: { description: "Internal Server Error." },
    },
};
async function handler(data) {
    var _a, _b, _c, _d, _e, _f, _g;
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    // Validate required fields
    if (!((_a = body.locationSettings) === null || _a === void 0 ? void 0 : _a.country)) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Location country is required for P2P offers",
        });
    }
    if (!((_c = (_b = body.tradeSettings) === null || _b === void 0 ? void 0 : _b.termsOfTrade) === null || _c === void 0 ? void 0 : _c.trim())) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Trade terms are required for P2P offers",
        });
    }
    // Log the incoming data for debugging
    console.log('[P2P Offer Create] Received body:', {
        type: body.type,
        currency: body.currency,
        paymentMethodIds: body.paymentMethodIds,
    });
    if (!body.paymentMethodIds || !Array.isArray(body.paymentMethodIds) || body.paymentMethodIds.length === 0) {
        console.error('[P2P Offer Create] Invalid payment methods:', body.paymentMethodIds);
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "At least one payment method is required for P2P offers",
        });
    }
    // For SELL offers, validate user has sufficient balance
    if (body.type === "SELL") {
        const requiredAmount = ((_d = body.amountConfig) === null || _d === void 0 ? void 0 : _d.total) || 0;
        if (requiredAmount <= 0) {
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "Invalid amount specified for sell offer",
            });
        }
        try {
            const wallet = await (0, utils_1.getWalletSafe)(user.id, body.walletType, body.currency);
            if (!wallet) {
                throw (0, error_1.createError)({
                    statusCode: 400,
                    message: `No ${body.walletType} wallet found for ${body.currency}`,
                });
            }
            const availableBalance = wallet.balance - wallet.inOrder;
            if (availableBalance < requiredAmount) {
                throw (0, error_1.createError)({
                    statusCode: 400,
                    message: `Insufficient balance. Available: ${availableBalance} ${body.currency}, Required: ${requiredAmount} ${body.currency}`,
                });
            }
        }
        catch (error) {
            // If it's already a createError, rethrow it
            if (error.statusCode) {
                throw error;
            }
            // Otherwise, wrap it in a generic error
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "Unable to verify wallet balance",
            });
        }
    }
    // Check if auto-approval is enabled
    const cacheManager = cache_1.CacheManager.getInstance();
    const autoApprove = await cacheManager.getSetting("p2pAutoApproveOffers");
    const shouldAutoApprove = autoApprove === true || autoApprove === "true";
    // start a transaction so creation + associations roll back together
    const t = await db_1.sequelize.transaction();
    try {
        // 1. create the offer
        const offer = await db_1.models.p2pOffer.create({
            userId: user.id,
            type: body.type,
            currency: body.currency,
            walletType: body.walletType,
            amountConfig: body.amountConfig,
            priceConfig: body.priceConfig,
            tradeSettings: body.tradeSettings,
            locationSettings: (_e = body.locationSettings) !== null && _e !== void 0 ? _e : null,
            userRequirements: (_f = body.userRequirements) !== null && _f !== void 0 ? _f : null,
            status: shouldAutoApprove ? "ACTIVE" : "PENDING_APPROVAL",
            views: 0,
            systemTags: [],
            adminNotes: null,
        }, { transaction: t });
        // 2. if any paymentMethodIds provided, validate & associate
        const ids = Array.isArray(body.paymentMethodIds)
            ? body.paymentMethodIds
            : [];
        if (ids.length) {
            console.log('[P2P Offer Create] Validating payment method IDs:', ids);
            // fetch and ensure all exist - also check for user-created methods
            const methods = await db_1.models.p2pPaymentMethod.findAll({
                where: {
                    id: ids,
                    [sequelize_1.Op.or]: [
                        { userId: null }, // System payment methods
                        { userId: user.id } // User's custom payment methods
                    ]
                },
                transaction: t,
            });
            console.log('[P2P Offer Create] Found payment methods:', methods.map(m => ({ id: m.id, name: m.name, userId: m.userId })));
            if (methods.length !== ids.length) {
                const foundIds = methods.map(m => m.id);
                const missingIds = ids.filter(id => !foundIds.includes(id));
                console.error('[P2P Offer Create] Missing payment method IDs:', missingIds);
                throw (0, error_1.createError)({
                    statusCode: 400,
                    message: `Invalid payment method IDs: ${missingIds.join(', ')}. Please ensure all payment methods are properly created first.`,
                });
            }
            // set the M:N association
            await offer.setPaymentMethods(methods, { transaction: t });
            console.log('[P2P Offer Create] Associated payment methods successfully');
        }
        // commit everything
        await t.commit();
        // reload to include the paymentMethods in the response
        await offer.reload({
            include: [
                {
                    model: db_1.models.p2pPaymentMethod,
                    as: "paymentMethods",
                    attributes: ["id", "name", "icon"],
                    through: { attributes: [] },
                }
            ]
        });
        return { message: "Offer created successfully.", offer };
    }
    catch (err) {
        await t.rollback();
        throw (0, error_1.createError)({
            statusCode: (_g = err.statusCode) !== null && _g !== void 0 ? _g : 500,
            message: err.message
                ? `Internal Server Error: ${err.message}`
                : "Internal Server Error",
        });
    }
}
