"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const query_1 = require("@b/utils/query");
const sequelize_1 = require("sequelize");
exports.metadata = {
    summary: "Get Trade Dashboard Data",
    description: "Retrieves aggregated trade data for the authenticated user.",
    operationId: "getP2PTradeDashboardData",
    tags: ["P2P", "Trade"],
    responses: {
        200: { description: "Trade dashboard data retrieved successfully." },
        401: query_1.unauthorizedResponse,
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    const { user } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id))
        throw new Error("Unauthorized");
    try {
        // ------ 1. TRADE STATS ------
        const [totalTrades, completedTrades, disputedTrades, activeTrades, pendingTrades, trades, recentActivity,] = await Promise.all([
            db_1.models.p2pTrade.count({
                where: { [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }] },
            }),
            db_1.models.p2pTrade.count({
                where: {
                    status: "COMPLETED",
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
            }),
            db_1.models.p2pTrade.findAll({
                where: {
                    status: "DISPUTED",
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
                limit: 7,
                order: [["updatedAt", "DESC"]],
                raw: true,
            }),
            db_1.models.p2pTrade.findAll({
                where: {
                    status: { [sequelize_1.Op.in]: ["IN_PROGRESS", "PENDING", "PAYMENT_SENT"] },
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
                order: [["updatedAt", "DESC"]],
                raw: true,
            }),
            db_1.models.p2pTrade.findAll({
                where: {
                    status: "PENDING",
                    [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }],
                },
                order: [["createdAt", "DESC"]],
                raw: true,
            }),
            // For calculating stats and volume
            db_1.models.p2pTrade.findAll({
                where: { [sequelize_1.Op.or]: [{ buyerId: user.id }, { sellerId: user.id }] },
                raw: true,
            }),
            db_1.models.p2pActivityLog.findAll({
                where: {
                    userId: user.id,
                    relatedEntity: "TRADE" // Only fetch trade-related activities
                },
                order: [["createdAt", "DESC"]],
                limit: 5,
                raw: true,
            }),
        ]);
        // ------ 2. Calculations ------
        const totalVolume = trades.reduce((sum, t) => sum + (t.fiatAmount || 0), 0);
        const avgCompletionTime = (() => {
            const completed = trades.filter((t) => t.status === "COMPLETED" && t.completedAt && t.createdAt);
            if (!completed.length)
                return null;
            const totalMs = completed.reduce((sum, t) => sum +
                (new Date(t.completedAt).getTime() - new Date(t.createdAt).getTime()), 0);
            const avgMs = totalMs / completed.length;
            // Format to h:mm:ss or similar
            const hours = Math.floor(avgMs / 3600000);
            const minutes = Math.floor((avgMs % 3600000) / 60000);
            return hours ? `${hours}h ${minutes}m` : `${minutes}m`;
        })();
        const successRate = totalTrades
            ? Math.round((completedTrades / totalTrades) * 100)
            : 0;
        // ------ 3. Helper for getting counterparty ------
        const getCounterparty = (trade) => {
            return trade.buyerId === user.id
                ? trade.sellerName || `User #${trade.sellerId}`
                : trade.buyerName || `User #${trade.buyerId}`;
        };
        // ------ 4. Format trades for frontend ------
        function formatTrade(trade) {
            return {
                id: trade.id,
                type: trade.buyerId === user.id ? "BUY" : "SELL",
                coin: trade.coin || trade.crypto || "N/A",
                amount: trade.amount,
                fiatAmount: trade.fiatAmount,
                price: trade.price,
                counterparty: getCounterparty(trade),
                status: trade.status,
                date: trade.updatedAt || trade.createdAt,
                paymentMethod: trade.paymentMethod || null,
            };
        }
        // ------ 5. Format activity log ------
        function formatActivity(act) {
            let message = act.message || act.details;
            // Try to parse JSON message and format it
            if (typeof message === 'string') {
                try {
                    const parsed = JSON.parse(message);
                    // Format based on the data structure
                    if (parsed.offerType && parsed.currency) {
                        const action = parsed.statusChange ? `Status changed from ${parsed.statusChange}` : 'Offer updated';
                        const updater = parsed.updatedBy || 'System';
                        message = `${action} for ${parsed.offerType} ${parsed.currency} by ${updater}`;
                    }
                    else if (parsed.action) {
                        message = parsed.action;
                    }
                    else if (parsed.description) {
                        message = parsed.description;
                    }
                    else {
                        // If we can't format it nicely, use a generic message
                        message = 'Activity recorded';
                    }
                }
                catch (e) {
                    // If it's not JSON or parsing fails, use the original message
                    // But if it starts with { or [, it's likely broken JSON - use generic message
                    if (message.trim().startsWith('{') || message.trim().startsWith('[')) {
                        message = 'Activity recorded';
                    }
                }
            }
            return {
                id: act.id,
                type: act.type || act.activityType,
                tradeId: act.tradeId,
                message: message,
                time: act.createdAt,
            };
        }
        // ------ 6. Prepare response ------
        return {
            tradeStats: {
                activeCount: activeTrades.length,
                completedCount: completedTrades,
                totalVolume,
                avgCompletionTime,
                successRate,
            },
            recentActivity: recentActivity.map(formatActivity),
            activeTrades: activeTrades.map(formatTrade),
            pendingTrades: pendingTrades.map(formatTrade),
            completedTrades: trades
                .filter((t) => t.status === "COMPLETED")
                .sort((a, b) => new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime())
                .slice(0, 7)
                .map(formatTrade),
            disputedTrades: disputedTrades.map(formatTrade),
        };
    }
    catch (err) {
        throw new Error("Internal Server Error: " + err.message);
    }
};
