"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.P2PRiskLevel = exports.P2PAuditEventType = void 0;
exports.createP2PAuditLog = createP2PAuditLog;
exports.createP2PAuditLogBatch = createP2PAuditLogBatch;
exports.withAuditLog = withAuditLog;
exports.getP2PAuditLogs = getP2PAuditLogs;
exports.exportP2PAuditLogs = exportP2PAuditLogs;
const db_1 = require("@b/db");
/**
 * Audit event types for P2P operations
 */
var P2PAuditEventType;
(function (P2PAuditEventType) {
    // Trade events
    P2PAuditEventType["TRADE_INITIATED"] = "TRADE_INITIATED";
    P2PAuditEventType["TRADE_PAYMENT_CONFIRMED"] = "TRADE_PAYMENT_CONFIRMED";
    P2PAuditEventType["TRADE_FUNDS_RELEASED"] = "TRADE_FUNDS_RELEASED";
    P2PAuditEventType["TRADE_CANCELLED"] = "TRADE_CANCELLED";
    P2PAuditEventType["TRADE_DISPUTED"] = "TRADE_DISPUTED";
    P2PAuditEventType["TRADE_COMPLETED"] = "TRADE_COMPLETED";
    P2PAuditEventType["TRADE_EXPIRED"] = "TRADE_EXPIRED";
    // Offer events
    P2PAuditEventType["OFFER_CREATED"] = "OFFER_CREATED";
    P2PAuditEventType["OFFER_UPDATED"] = "OFFER_UPDATED";
    P2PAuditEventType["OFFER_DELETED"] = "OFFER_DELETED";
    P2PAuditEventType["OFFER_PAUSED"] = "OFFER_PAUSED";
    P2PAuditEventType["OFFER_ACTIVATED"] = "OFFER_ACTIVATED";
    // Financial events
    P2PAuditEventType["FUNDS_LOCKED"] = "FUNDS_LOCKED";
    P2PAuditEventType["FUNDS_UNLOCKED"] = "FUNDS_UNLOCKED";
    P2PAuditEventType["FUNDS_TRANSFERRED"] = "FUNDS_TRANSFERRED";
    P2PAuditEventType["FEE_CHARGED"] = "FEE_CHARGED";
    // Admin events
    P2PAuditEventType["ADMIN_TRADE_RESOLVED"] = "ADMIN_TRADE_RESOLVED";
    P2PAuditEventType["ADMIN_DISPUTE_RESOLVED"] = "ADMIN_DISPUTE_RESOLVED";
    P2PAuditEventType["ADMIN_OFFER_APPROVED"] = "ADMIN_OFFER_APPROVED";
    P2PAuditEventType["ADMIN_USER_BANNED"] = "ADMIN_USER_BANNED";
    // Security events
    P2PAuditEventType["SUSPICIOUS_ACTIVITY"] = "SUSPICIOUS_ACTIVITY";
    P2PAuditEventType["RATE_LIMIT_EXCEEDED"] = "RATE_LIMIT_EXCEEDED";
    P2PAuditEventType["UNAUTHORIZED_ACCESS"] = "UNAUTHORIZED_ACCESS";
    P2PAuditEventType["VALIDATION_FAILED"] = "VALIDATION_FAILED";
})(P2PAuditEventType || (exports.P2PAuditEventType = P2PAuditEventType = {}));
/**
 * Risk levels for audit events
 */
var P2PRiskLevel;
(function (P2PRiskLevel) {
    P2PRiskLevel["LOW"] = "LOW";
    P2PRiskLevel["MEDIUM"] = "MEDIUM";
    P2PRiskLevel["HIGH"] = "HIGH";
    P2PRiskLevel["CRITICAL"] = "CRITICAL";
})(P2PRiskLevel || (exports.P2PRiskLevel = P2PRiskLevel = {}));
/**
 * Create a comprehensive audit log entry
 */
async function createP2PAuditLog(log) {
    try {
        // Determine risk level if not provided
        const riskLevel = log.riskLevel || determineRiskLevel(log.eventType, log.metadata);
        // Create the audit log entry with correct field names
        await db_1.models.p2pActivityLog.create({
            userId: log.userId,
            type: log.eventType,
            action: log.eventType, // Use eventType as action since it describes what happened
            relatedEntity: log.entityType,
            relatedEntityId: log.entityId,
            details: JSON.stringify({
                ...log.metadata,
                timestamp: new Date().toISOString(),
                riskLevel,
                isAdminAction: log.isAdminAction || false,
                adminId: log.adminId,
            }),
        });
        // For high-risk events, create additional alert
        if (riskLevel === P2PRiskLevel.HIGH || riskLevel === P2PRiskLevel.CRITICAL) {
            await createSecurityAlert(log, riskLevel);
        }
    }
    catch (error) {
        console.error("Failed to create P2P audit log:", error);
        // Audit logging should not break the main flow
    }
}
/**
 * Create a batch of audit logs (for complex operations)
 */
async function createP2PAuditLogBatch(logs) {
    try {
        const auditEntries = logs.map(log => ({
            userId: log.userId,
            type: log.eventType,
            action: log.eventType, // Use eventType as action
            relatedEntity: log.entityType,
            relatedEntityId: log.entityId,
            details: JSON.stringify({
                ...log.metadata,
                timestamp: new Date().toISOString(),
                riskLevel: log.riskLevel || determineRiskLevel(log.eventType, log.metadata),
                isAdminAction: log.isAdminAction || false,
                adminId: log.adminId,
            }),
        }));
        await db_1.models.p2pActivityLog.bulkCreate(auditEntries);
    }
    catch (error) {
        console.error("Failed to create P2P audit log batch:", error);
    }
}
/**
 * Determine risk level based on event type and metadata
 */
function determineRiskLevel(eventType, metadata) {
    // Critical risk events
    if ([
        P2PAuditEventType.FUNDS_TRANSFERRED,
        P2PAuditEventType.UNAUTHORIZED_ACCESS,
        P2PAuditEventType.ADMIN_USER_BANNED,
    ].includes(eventType)) {
        return P2PRiskLevel.CRITICAL;
    }
    // High risk events
    if ([
        P2PAuditEventType.TRADE_DISPUTED,
        P2PAuditEventType.SUSPICIOUS_ACTIVITY,
        P2PAuditEventType.ADMIN_TRADE_RESOLVED,
        P2PAuditEventType.ADMIN_DISPUTE_RESOLVED,
    ].includes(eventType)) {
        return P2PRiskLevel.HIGH;
    }
    // Medium risk based on amount
    if (metadata.amount && metadata.amount > 1000) {
        return P2PRiskLevel.MEDIUM;
    }
    // Medium risk events
    if ([
        P2PAuditEventType.TRADE_CANCELLED,
        P2PAuditEventType.OFFER_DELETED,
        P2PAuditEventType.RATE_LIMIT_EXCEEDED,
    ].includes(eventType)) {
        return P2PRiskLevel.MEDIUM;
    }
    return P2PRiskLevel.LOW;
}
/**
 * Create security alert for high-risk events
 */
async function createSecurityAlert(log, riskLevel) {
    try {
        // Create notification for admins
        // const { notifyAdmins } = await import("./notifications");
        // TODO: Implement notifyAdmins function
        // TODO: Implement admin notification
        // await notifyAdmins("P2P_SECURITY_ALERT", {
        //   eventType: log.eventType,
        //   entityType: log.entityType,
        //   entityId: log.entityId,
        //   userId: log.userId,
        //   adminId: log.adminId,
        //   riskLevel,
        //   metadata: log.metadata,
        //   timestamp: new Date().toISOString(),
        // });
        // Log to security monitoring system
        console.warn("P2P Security Alert:", {
            eventType: log.eventType,
            riskLevel,
            userId: log.userId,
            entityId: log.entityId,
        });
    }
    catch (error) {
        console.error("Failed to create security alert:", error);
    }
}
/**
 * Audit wrapper for critical operations
 */
function withAuditLog(fn, eventType, entityType) {
    return (async (...args) => {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        const startTime = Date.now();
        let result;
        let error;
        try {
            result = await fn(...args);
            // Extract audit information from function context
            const context = args[0]; // Assuming first arg is the request context
            if (((_a = context === null || context === void 0 ? void 0 : context.user) === null || _a === void 0 ? void 0 : _a.id) && ((_b = context === null || context === void 0 ? void 0 : context.params) === null || _b === void 0 ? void 0 : _b.id)) {
                await createP2PAuditLog({
                    userId: context.user.id,
                    eventType,
                    entityType,
                    entityId: context.params.id,
                    metadata: {
                        success: true,
                        executionTime: Date.now() - startTime,
                        ip: context.ip || ((_c = context.connection) === null || _c === void 0 ? void 0 : _c.remoteAddress),
                        userAgent: (_d = context.headers) === null || _d === void 0 ? void 0 : _d["user-agent"],
                    },
                });
            }
            return result;
        }
        catch (err) {
            error = err;
            // Log failed attempts
            const context = args[0];
            if (((_e = context === null || context === void 0 ? void 0 : context.user) === null || _e === void 0 ? void 0 : _e.id) && ((_f = context === null || context === void 0 ? void 0 : context.params) === null || _f === void 0 ? void 0 : _f.id)) {
                await createP2PAuditLog({
                    userId: context.user.id,
                    eventType: P2PAuditEventType.VALIDATION_FAILED,
                    entityType,
                    entityId: context.params.id,
                    metadata: {
                        success: false,
                        error: err.message,
                        originalEvent: eventType,
                        executionTime: Date.now() - startTime,
                        ip: context.ip || ((_g = context.connection) === null || _g === void 0 ? void 0 : _g.remoteAddress),
                        userAgent: (_h = context.headers) === null || _h === void 0 ? void 0 : _h["user-agent"],
                    },
                    riskLevel: P2PRiskLevel.HIGH,
                });
            }
            throw error;
        }
    });
}
/**
 * Get audit logs for an entity
 */
async function getP2PAuditLogs(entityType, entityId, options) {
    var _a;
    const where = {
        relatedEntity: entityType,
        relatedEntityId: entityId,
    };
    if ((options === null || options === void 0 ? void 0 : options.startDate) || (options === null || options === void 0 ? void 0 : options.endDate)) {
        where.createdAt = {};
        if (options.startDate)
            where.createdAt.$gte = options.startDate;
        if (options.endDate)
            where.createdAt.$lte = options.endDate;
    }
    if ((_a = options === null || options === void 0 ? void 0 : options.eventTypes) === null || _a === void 0 ? void 0 : _a.length) {
        where.type = { $in: options.eventTypes };
    }
    return db_1.models.p2pActivityLog.findAll({
        where,
        limit: (options === null || options === void 0 ? void 0 : options.limit) || 100,
        offset: (options === null || options === void 0 ? void 0 : options.offset) || 0,
        order: [["createdAt", "DESC"]],
        include: [{
                model: db_1.models.user,
                as: "user",
                attributes: ["id", "firstName", "lastName", "email"],
            }],
    });
}
/**
 * Export audit logs for compliance
 */
async function exportP2PAuditLogs(startDate, endDate, options) {
    var _a, _b, _c;
    const where = {
        createdAt: {
            $gte: startDate,
            $lte: endDate,
        },
    };
    if ((_a = options === null || options === void 0 ? void 0 : options.entityTypes) === null || _a === void 0 ? void 0 : _a.length) {
        where.relatedEntity = { $in: options.entityTypes };
    }
    if ((_b = options === null || options === void 0 ? void 0 : options.eventTypes) === null || _b === void 0 ? void 0 : _b.length) {
        where.type = { $in: options.eventTypes };
    }
    if ((_c = options === null || options === void 0 ? void 0 : options.userIds) === null || _c === void 0 ? void 0 : _c.length) {
        where.userId = { $in: options.userIds };
    }
    const logs = await db_1.models.p2pActivityLog.findAll({
        where,
        order: [["createdAt", "ASC"]],
        include: [{
                model: db_1.models.user,
                as: "user",
                attributes: ["id", "firstName", "lastName", "email"],
            }],
    });
    return logs.map(log => {
        var _a;
        return ({
            id: log.id,
            timestamp: log.createdAt,
            userId: log.userId,
            userName: log.user ? `${log.user.firstName} ${log.user.lastName}` : "Unknown",
            userEmail: (_a = log.user) === null || _a === void 0 ? void 0 : _a.email,
            eventType: log.type,
            entityType: log.relatedEntity,
            entityId: log.relatedEntityId,
            metadata: log.details ? JSON.parse(log.details) : {},
            riskLevel: log.details ? JSON.parse(log.details).riskLevel : undefined,
            isAdminAction: log.details ? JSON.parse(log.details).isAdminAction : false,
            adminId: log.details ? JSON.parse(log.details).adminId : undefined,
        });
    });
}
