"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isOfferOwner = isOfferOwner;
exports.isTradeParticipant = isTradeParticipant;
exports.isPaymentMethodOwner = isPaymentMethodOwner;
exports.requireOfferOwnership = requireOfferOwnership;
exports.requireTradeParticipation = requireTradeParticipation;
exports.logP2PAdminAction = logP2PAdminAction;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
/**
 * Check if user owns a P2P offer
 */
async function isOfferOwner(userId, offerId) {
    const offer = await db_1.models.p2pOffer.findByPk(offerId, {
        attributes: ["userId"],
    });
    return (offer === null || offer === void 0 ? void 0 : offer.userId) === userId;
}
/**
 * Check if user is part of a P2P trade (buyer or seller)
 */
async function isTradeParticipant(userId, tradeId) {
    const trade = await db_1.models.p2pTrade.findByPk(tradeId, {
        attributes: ["buyerId", "sellerId"],
    });
    return (trade === null || trade === void 0 ? void 0 : trade.buyerId) === userId || (trade === null || trade === void 0 ? void 0 : trade.sellerId) === userId;
}
/**
 * Check if user owns a payment method
 */
async function isPaymentMethodOwner(userId, paymentMethodId) {
    const paymentMethod = await db_1.models.p2pPaymentMethod.findByPk(paymentMethodId, {
        attributes: ["userId"],
    });
    return (paymentMethod === null || paymentMethod === void 0 ? void 0 : paymentMethod.userId) === userId;
}
/**
 * Require offer ownership or throw error
 */
async function requireOfferOwnership(userId, offerId) {
    const isOwner = await isOfferOwner(userId, offerId);
    if (!isOwner) {
        throw (0, error_1.createError)({
            statusCode: 403,
            message: "You don't have permission to modify this offer",
        });
    }
}
/**
 * Require trade participation or throw error
 */
async function requireTradeParticipation(userId, tradeId) {
    const isParticipant = await isTradeParticipant(userId, tradeId);
    if (!isParticipant) {
        throw (0, error_1.createError)({
            statusCode: 403,
            message: "You are not part of this trade",
        });
    }
}
/**
 * Log P2P admin action for audit trail
 */
async function logP2PAdminAction(userId, action, entityType, entityId, metadata) {
    try {
        await db_1.models.p2pActivityLog.create({
            userId,
            type: `ADMIN_${action}`,
            action: action,
            relatedEntity: entityType,
            relatedEntityId: entityId,
            details: JSON.stringify({
                ...metadata,
                timestamp: new Date().toISOString(),
                isAdminAction: true,
            }),
        });
    }
    catch (error) {
        console.error("Failed to log P2P admin action:", error);
    }
}
