"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const sequelize_1 = require("sequelize");
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Creates a payment intent",
    operationId: "createPaymentIntent",
    tags: ["Payments"],
    requiresAuth: false, // No login required
    requiresApi: true, // API key required
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        amount: { type: "number", description: "Amount to be paid" },
                        currency: { type: "string", description: "Payment currency" },
                        successUrl: {
                            type: "string",
                            description: "URL for successful payment",
                        },
                        failUrl: { type: "string", description: "URL for failed payment" },
                        description: {
                            type: "string",
                            description: "Optional description for the payment intent",
                        },
                        products: {
                            type: "array",
                            items: {
                                type: "object",
                                properties: {
                                    name: { type: "string", description: "Product name" },
                                    quantity: { type: "number", description: "Quantity" },
                                    price: { type: "number", description: "Price" },
                                    currency: { type: "string", description: "Product currency" },
                                    sku: { type: "string", description: "Product SKU" },
                                },
                                required: ["name", "quantity", "price", "currency"],
                            },
                        },
                    },
                    required: ["amount", "currency", "successUrl", "failUrl", "products"],
                },
            },
        },
    },
    responses: {
        200: { description: "Payment intent created successfully" },
        400: { description: "Invalid request" },
    },
};
exports.default = async (data) => {
    const { body, headers } = data;
    const apiKey = headers["x-api-key"];
    if (!apiKey) {
        throw (0, error_1.createError)({
            statusCode: 401,
            message: "API key is required",
        });
    }
    const { amount, currency, ipnUrl, successUrl, failUrl, description, products, tax, discount, } = body;
    if (!amount || amount <= 0) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Invalid amount",
        });
    }
    if (!currency) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Currency is required",
        });
    }
    if (!ipnUrl) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "IPN URL is required",
        });
    }
    if (!successUrl) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Success URL is required",
        });
    }
    if (!failUrl) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "Fail URL is required",
        });
    }
    if (!products || !products.length) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: "At least one product is required",
        });
    }
    // Cleanup stale intents
    const oneDayAgo = new Date();
    oneDayAgo.setDate(oneDayAgo.getDate() - 1);
    await db_1.models.paymentIntent.destroy({
        where: {
            [sequelize_1.Op.or]: [
                { status: "PENDING", createdAt: { [sequelize_1.Op.lt]: oneDayAgo } }, // Expired intents
            ],
        },
        individualHooks: true,
    });
    // Create the payment intent
    const paymentIntent = await db_1.models.paymentIntent.create({
        amount,
        currency,
        tax: tax || 0,
        discount: discount || 0,
        status: "PENDING",
        ipnUrl,
        successUrl,
        failUrl,
        apiKey,
        description: description || null,
        transactionId: null,
    });
    // Store product details with images
    await db_1.models.paymentIntentProduct.bulkCreate(products.map((product) => ({
        paymentIntentId: paymentIntent.id,
        name: product.name,
        quantity: product.quantity,
        price: product.price,
        currency: product.currency,
        sku: product.sku,
        image: product.image, // Add image to the model
    })));
    return {
        paymentIntentId: paymentIntent.id,
        status: "created",
        currency,
    };
};
