"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const query_1 = require("@b/utils/query");
exports.metadata = {
    summary: "Import Binary Markets from Exchange Markets",
    operationId: "importBinaryMarkets",
    tags: ["Admin", "Binary", "Markets"],
    description: "Imports spot markets from exchange markets and creates binary markets for trading.",
    requiresAuth: true,
    responses: {
        200: {
            description: "Binary markets imported successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: { type: "string" },
                            imported: { type: "number" },
                            skipped: { type: "number" },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Exchange Markets"),
        500: query_1.serverErrorResponse,
    },
    permission: "create.binary.market",
};
exports.default = async (data) => {
    try {
        // Get all active exchange markets (spot markets only)
        const exchangeMarkets = await db_1.models.exchangeMarket.findAll({
            where: {
                status: true,
            },
            attributes: ["currency", "pair"],
        });
        if (exchangeMarkets.length === 0) {
            throw new Error("No active exchange markets found to import");
        }
        // Get existing binary markets to avoid duplicates
        const existingBinaryMarkets = await db_1.models.binaryMarket.findAll({
            attributes: ["currency", "pair"],
        });
        const existingPairs = new Set(existingBinaryMarkets.map((m) => `${m.currency}/${m.pair}`));
        // Filter out markets that already exist in binary markets
        const marketsToImport = exchangeMarkets.filter((market) => !existingPairs.has(`${market.currency}/${market.pair}`));
        let imported = 0;
        const skipped = exchangeMarkets.length - marketsToImport.length;
        // Import markets in transaction
        await db_1.sequelize.transaction(async (transaction) => {
            for (const market of marketsToImport) {
                await db_1.models.binaryMarket.create({
                    currency: market.currency,
                    pair: market.pair,
                    isTrending: false,
                    isHot: false,
                    status: false, // Import as disabled by default
                }, { transaction });
                imported++;
            }
        });
        return {
            message: `Successfully imported ${imported} binary markets from exchange markets`,
            imported,
            skipped,
        };
    }
    catch (error) {
        console.error("Error importing binary markets:", error);
        throw error;
    }
};
