"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const db_1 = require("@b/db");
const sequelize_1 = require("sequelize");
const utils_1 = require("@b/api/admin/system/utils");
exports.metadata = {
    summary: "Lists all comments with pagination and optional filtering by user or post",
    operationId: "listComments",
    tags: ["Admin", "Content", "Comment"],
    responses: {
        200: {
            description: "List of comments with user and post details and pagination",
            content: {
                "application/json": {
                    schema: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                id: { type: "string", description: "ID of the extension" },
                                productId: {
                                    type: "string",
                                    description: "Product ID of the extension",
                                },
                                name: { type: "string", description: "Name of the extension" },
                                title: {
                                    type: "string",
                                    description: "Title of the extension",
                                },
                                description: {
                                    type: "string",
                                    description: "Description of the extension",
                                },
                                link: { type: "string", description: "Link to the extension" },
                                status: {
                                    type: "boolean",
                                    description: "Status of the extension",
                                },
                                version: {
                                    type: "string",
                                    description: "Version of the extension",
                                },
                                image: {
                                    type: "string",
                                    description: "Image of the extension",
                                },
                                hasLicenseUpdate: {
                                    type: "boolean",
                                    description: "Whether the extension has an update available in the license system",
                                },
                                licenseVersion: {
                                    type: "string",
                                    description: "Latest version available in the license system",
                                },
                                licenseReleaseDate: {
                                    type: "string",
                                    description: "Release date of the latest version in the license system",
                                },
                                licenseSummary: {
                                    type: "string",
                                    description: "Summary of the latest version in the license system",
                                },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Extensions"),
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
    permission: "view.extension",
};
exports.default = async (data) => {
    const extensions = await db_1.models.extension.findAll({
        where: { [sequelize_1.Op.not]: { name: "swap" } },
        attributes: {
            exclude: ["createdAt", "updatedAt"],
        },
    });
    // Fetch license updates
    let licenseUpdates = { status: false, products: [] };
    try {
        licenseUpdates = await (0, utils_1.fetchAllProductsUpdates)();
    }
    catch (error) {
        console.error('Failed to fetch license updates:', error);
    }
    // Map license updates to extensions
    const extensionsWithUpdates = extensions.map((extension) => {
        var _a;
        const extensionData = extension.toJSON();
        // Find corresponding license product
        const licenseProduct = (_a = licenseUpdates.products) === null || _a === void 0 ? void 0 : _a.find((product) => product.product_id === extension.productId);
        // Add update information
        extensionData.hasLicenseUpdate = false;
        extensionData.licenseVersion = null;
        extensionData.licenseReleaseDate = null;
        extensionData.licenseSummary = null;
        if (licenseProduct && licenseProduct.has_version) {
            extensionData.licenseVersion = licenseProduct.latest_version;
            extensionData.licenseReleaseDate = licenseProduct.release_date;
            extensionData.licenseSummary = licenseProduct.summary;
            // Check if there's a newer version available
            if (licenseProduct.latest_version && extension.version) {
                // Simple version comparison (you might want to use a proper semver library)
                const currentVersion = extension.version.split('.').map(Number);
                const latestVersion = licenseProduct.latest_version.split('.').map(Number);
                for (let i = 0; i < Math.max(currentVersion.length, latestVersion.length); i++) {
                    const current = currentVersion[i] || 0;
                    const latest = latestVersion[i] || 0;
                    if (latest > current) {
                        extensionData.hasLicenseUpdate = true;
                        break;
                    }
                    else if (current > latest) {
                        break;
                    }
                }
            }
        }
        return extensionData;
    });
    return extensionsWithUpdates;
};
