"use strict";
// File: /server/api/admin/logs/index.get.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
exports.getFilteredLogs = getFilteredLogs;
const constants_1 = require("@b/utils/constants");
const fs_1 = require("fs");
const path_1 = require("path");
const utils_1 = require("./utils");
const validation_1 = require("@b/utils/validation");
exports.metadata = {
    summary: "Fetches log files based on category and date",
    operationId: "fetchLogFiles",
    tags: ["Admin", "Logs"],
    parameters: constants_1.crudParameters,
    responses: {
        200: {
            description: "Log entries for the given category and date",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            items: {
                                type: "array",
                                items: { type: "object", additionalProperties: true },
                            },
                            pagination: {
                                type: "object",
                                properties: {
                                    totalItems: { type: "number" },
                                    currentPage: { type: "number" },
                                    perPage: { type: "number" },
                                    totalPages: { type: "number" },
                                },
                            },
                        },
                    },
                },
            },
        },
        400: {
            description: "Bad request if the category or date are not specified",
        },
        404: { description: "Not found if the log file does not exist" },
        500: { description: "Internal server error" },
    },
    requiresAuth: true,
    permission: "view.system.log",
};
exports.default = async (data) => {
    const { query } = data;
    return getFilteredLogs(query);
};
/**
 * Reads a log file based on a date (defaulting to today), then applies filtering,
 * sorting (by a specified field and order), and pagination to return a structured response.
 */
async function getFilteredLogs(query) {
    var _a;
    // Pagination defaults
    const page = query.page ? parseInt(query.page) : 1;
    const perPage = query.perPage ? parseInt(query.perPage) : 10;
    const sortField = query.sortField || "timestamp";
    const sortOrder = query.sortOrder || "asc";
    // Parse filter parameter if provided
    let filters = {};
    try {
        filters = query.filter ? JSON.parse(query.filter) : {};
    }
    catch (error) {
        console.error("Error parsing filter:", error);
    }
    // Use a 'date' filter (if provided) to choose the log file; default to today's date.
    const date = ((_a = filters.date) === null || _a === void 0 ? void 0 : _a.value) || new Date().toISOString().split("T")[0];
    delete filters.date; // Remove date from filters so it isn't applied on log entries
    // Sanitize the date to prevent local file inclusion vulnerabilities
    const sanitizedDate = (0, validation_1.sanitizePath)(date);
    const logFilePath = (0, path_1.join)(process.cwd(), "logs", `${sanitizedDate}.log`);
    let fileData;
    try {
        fileData = await fs_1.promises.readFile(logFilePath, { encoding: "utf8" });
    }
    catch (error) {
        console.error(`Error reading log file: ${error.message}`);
        if (error.code === "ENOENT") {
            return {
                items: [],
                pagination: {
                    totalItems: 0,
                    currentPage: page,
                    perPage,
                    totalPages: 0,
                },
            };
        }
        throw error;
    }
    // Split file data into individual lines and parse each line as JSON.
    // Each log entry gets an 'id' based on its index.
    let logs = fileData
        .split("\n")
        .filter((line) => line.trim())
        .map((line, index) => {
        try {
            const parsedLine = JSON.parse(line);
            return { id: index.toString(), ...parsedLine };
        }
        catch (parseError) {
            console.error(`Error parsing log line: ${line}`);
            return null;
        }
    })
        .filter(Boolean);
    // Apply filters using the operator functions from operatorMap.
    logs = logs.filter((log) => Object.entries(filters).every(([key, filter]) => {
        let filterValue, operator;
        if (typeof filter === "object" && filter.operator) {
            operator = filter.operator;
            filterValue = filter.value;
        }
        else {
            operator = "equal";
            filterValue = filter;
        }
        const operation = utils_1.operatorMap[operator];
        return typeof operation === "function"
            ? operation(log, key, filterValue)
            : log[key] == filterValue;
    }));
    // Sort logs by the provided sortField and sortOrder.
    logs.sort((a, b) => {
        const aVal = a[sortField];
        const bVal = b[sortField];
        if (aVal < bVal)
            return sortOrder === "asc" ? -1 : 1;
        if (aVal > bVal)
            return sortOrder === "asc" ? 1 : -1;
        return 0;
    });
    // Paginate the results.
    const totalItems = logs.length;
    const totalPages = Math.ceil(totalItems / perPage);
    const offset = (page - 1) * perPage;
    const paginatedItems = logs.slice(offset, offset + perPage);
    return {
        items: paginatedItems,
        pagination: {
            totalItems,
            currentPage: page,
            perPage,
            totalPages,
        },
    };
}
