"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProduct = getProduct;
exports.getBlockchain = getBlockchain;
exports.fetchPublicIp = fetchPublicIp;
exports.getPublicIp = getPublicIp;
exports.callApi = callApi;
exports.verifyLicense = verifyLicense;
exports.activateLicense = activateLicense;
exports.checkLatestVersion = checkLatestVersion;
exports.checkUpdate = checkUpdate;
exports.downloadUpdate = downloadUpdate;
exports.fetchAllProductsUpdates = fetchAllProductsUpdates;
const https_1 = __importDefault(require("https"));
const adm_zip_1 = __importDefault(require("adm-zip"));
const fs_1 = require("fs");
const fs_2 = require("fs");
const system_1 = require("../../../utils/system");
const db_1 = require("@b/db");
const path_1 = __importDefault(require("path"));
const license_1 = require("@b/config/license");
// Secure admin error utility
function adminError(message = "System configuration error. Please contact administrator.", details) {
    if (details) {
        // Only log server-side, never return to user
        console.error("[ADMIN SYSTEM ERROR]", message, details);
    }
    else {
        console.error("[ADMIN SYSTEM ERROR]", message);
    }
    return new Error(message);
}
let cachedIP = null;
let lastFetched = null;
let nextVerificationDate = null;
const verificationPeriodDays = 3;
// Determine the correct root path based on environment
const rootPath = (() => {
    const cwd = process.cwd();
    // In development, if we're in the backend folder, go up one level
    if (cwd.endsWith('/backend') || cwd.endsWith('\\backend')) {
        return path_1.default.join(cwd, '..');
    }
    // In production or if already at root, use current directory
    return cwd;
})();
const licFolderPath = `${rootPath}/lic`;
async function getProduct(id) {
    if (id) {
        const extension = await db_1.models.extension.findOne({
            where: { productId: id },
        });
        if (!extension)
            throw adminError();
        return extension;
    }
    else {
        try {
            // Try multiple possible locations for package.json
            const possiblePaths = [
                `${rootPath}/package.json`,
                `${path_1.default.join(rootPath, '..')}/package.json`,
                `${process.cwd()}/package.json`,
                `${path_1.default.join(process.cwd(), '..')}/package.json`,
            ];
            let content = null;
            let usedPath = '';
            for (const filePath of possiblePaths) {
                try {
                    const fileContent = await fs_1.promises.readFile(filePath, "utf8");
                    content = JSON.parse(fileContent);
                    // Check if this package.json has the expected fields
                    if (content && (content.id || content.name)) {
                        usedPath = filePath;
                        break;
                    }
                }
                catch (err) {
                    continue;
                }
            }
            if (!content || !content.id) {
                throw adminError("Could not find valid package.json with required fields");
            }
            return {
                id: content.id || "7848B8AC", // Fallback to default product ID
                productId: content.id || "7848B8AC", // Map id to productId for compatibility
                name: content.name || "bicrypto",
                version: content.version || "5.0.0",
                description: content.description || "BiCrypto Trading Platform",
            };
        }
        catch (error) {
            throw adminError("Could not read product information.", error);
        }
    }
}
async function getBlockchain(id) {
    const blockchain = await db_1.models.ecosystemBlockchain.findOne({
        where: { productId: id },
    });
    if (!blockchain)
        throw adminError();
    return blockchain;
}
async function fetchPublicIp() {
    try {
        const data = await new Promise((resolve, reject) => {
            https_1.default.get("https://api.ipify.org?format=json", (resp) => {
                let data = "";
                resp.on("data", (chunk) => {
                    data += chunk;
                });
                resp.on("end", () => {
                    resolve(JSON.parse(data));
                });
                resp.on("error", (err) => {
                    reject(err);
                });
            });
        });
        return data.ip;
    }
    catch (error) {
        console.error(`[ADMIN SYSTEM ERROR] Error fetching public IP: ${error.message}`);
        return null;
    }
}
async function getPublicIp() {
    const now = Date.now();
    if (cachedIP && lastFetched && now - lastFetched < 60000) {
        // 1 minute cache
        return cachedIP;
    }
    cachedIP = await fetchPublicIp();
    lastFetched = now;
    return cachedIP;
}
async function callApi(method, url, data = null, filename) {
    try {
        const licenseConfig = (0, license_1.getLicenseConfig)();
        const headers = {
            "Content-Type": "application/json",
            "LB-API-KEY": process.env.API_LICENSE_API_KEY || licenseConfig.apiKey,
            "LB-URL": process.env.NEXT_PUBLIC_SITE_URL || "http://localhost:3000",
            "LB-IP": (await getPublicIp()) || "127.0.0.1",
            "LB-LANG": "en",
        };
        const requestData = data ? JSON.stringify(data) : null;
        const requestOptions = {
            method: method,
            headers: headers,
        };
        const response = await new Promise((resolve, reject) => {
            const req = https_1.default.request(url, requestOptions, (res) => {
                const data = [];
                if (res.headers["content-type"] === "application/zip") {
                    if (!filename) {
                        reject(adminError("Filename required for zip download."));
                        return;
                    }
                    const dirPath = `${rootPath}/updates`;
                    const filePath = `${dirPath}/${filename}.zip`;
                    // Ensure the directory exists
                    fs_1.promises.mkdir(dirPath, { recursive: true })
                        .then(() => {
                        const fileStream = (0, fs_2.createWriteStream)(filePath);
                        res.pipe(fileStream);
                        fileStream.on("finish", () => {
                            resolve({
                                status: true,
                                message: "Update file downloaded successfully",
                                path: filePath,
                            });
                        });
                        fileStream.on("error", (err) => {
                            reject(adminError("Download error.", err));
                        });
                    })
                        .catch((err) => {
                        reject(adminError("Directory error.", err));
                    });
                }
                else {
                    res.on("data", (chunk) => {
                        data.push(chunk);
                    });
                    res.on("end", () => {
                        let result;
                        try {
                            result = JSON.parse(Buffer.concat(data).toString());
                        }
                        catch (e) {
                            reject(adminError("Invalid response from server.", e));
                            return;
                        }
                        if (res.statusCode !== 200) {
                            reject(adminError("Operation failed.", result));
                        }
                        else {
                            resolve(result);
                        }
                    });
                }
                res.on("error", (err) => {
                    reject(adminError("Response error.", err));
                });
            });
            req.on("error", (err) => {
                reject(adminError("Request error.", err));
            });
            if (requestData) {
                req.write(requestData);
            }
            req.end();
        });
        return response;
    }
    catch (error) {
        throw adminError("API call failed.", error);
    }
}
async function verifyLicense(productId, license, client, timeBasedCheck) {
    const licenseFilePath = `${licFolderPath}/${productId}.lic`;
    let data;
    try {
        // Check if a license file exists
        const licenseFileContent = await fs_1.promises.readFile(licenseFilePath, "utf8");
        data = {
            product_id: productId,
            license_file: licenseFileContent,
            license_code: null,
            client_name: null,
        };
    }
    catch (err) {
        // File does not exist or other error occurred
        data = {
            product_id: productId,
            license_file: null,
            license_code: license,
            client_name: client,
        };
    }
    if (timeBasedCheck && verificationPeriodDays > 0) {
        const today = new Date();
        if (nextVerificationDate && today < nextVerificationDate) {
            return { status: true, message: "Verified from cache" };
        }
    }
    const licenseConfig = (0, license_1.getLicenseConfig)();
    const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
    const response = await callApi("POST", `${apiUrl}/api/verify_license`, data);
    if (timeBasedCheck && verificationPeriodDays > 0 && response.status) {
        const today = new Date();
        nextVerificationDate = new Date();
        nextVerificationDate.setDate(today.getDate() + verificationPeriodDays);
    }
    if (!response.status) {
        throw adminError("License verification failed.");
    }
    return response;
}
async function activateLicense(productId, license, client) {
    const data = {
        product_id: productId,
        license_code: license,
        client_name: client,
        verify_type: "envato",
    };
    const licenseConfig = (0, license_1.getLicenseConfig)();
    const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
    const response = await callApi("POST", `${apiUrl}/api/activate_license`, data);
    if (!response.status) {
        throw adminError("License activation failed.");
    }
    // If activation is successful, save the license
    if (response.lic_response) {
        const licFileContent = response.lic_response;
        const licenseFilePath = `${licFolderPath}/${productId}.lic`;
        // Ensure the lic directory exists
        await fs_1.promises.mkdir(licFolderPath, { recursive: true });
        // Save the license to a file in the lic directory
        await fs_1.promises.writeFile(licenseFilePath, licFileContent);
    }
    return response;
}
async function checkLatestVersion(productId) {
    const licenseConfig = (0, license_1.getLicenseConfig)();
    const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
    const payload = { product_id: productId };
    return await callApi("POST", `${apiUrl}/api/latest_version`, payload);
}
async function checkUpdate(productId, currentVersion) {
    const licenseConfig = (0, license_1.getLicenseConfig)();
    const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
    const payload = {
        product_id: productId,
        current_version: currentVersion,
    };
    return await callApi("POST", `${apiUrl}/api/check_update`, payload);
}
async function downloadUpdate(productId, updateId, version, product, type) {
    if (!productId || !updateId || !version || !product) {
        throw adminError();
    }
    const licenseFilePath = `${licFolderPath}/${productId}.lic`;
    let licenseFile;
    try {
        licenseFile = await fs_1.promises.readFile(licenseFilePath, "utf8");
    }
    catch (e) {
        throw adminError();
    }
    const data = {
        license_file: licenseFile,
        license_code: null,
        client_name: null,
    };
    // Call API to download update
    const licenseConfig = (0, license_1.getLicenseConfig)();
    const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
    const response = await callApi("POST", `${apiUrl}/api/download_update/main/${updateId}`, data, `${product}-${version}`);
    if (!response.status || !response.path) {
        throw adminError("Update download failed.");
    }
    try {
        unzip(response.path, rootPath);
        if (type === "extension") {
            try {
                await (0, system_1.updateExtensionQuery)(productId, version);
            }
            catch (error) {
                throw adminError("Extension update failed.", error);
            }
        }
        else if (type === "blockchain") {
            try {
                await (0, system_1.updateBlockchainQuery)(productId, version);
            }
            catch (error) {
                throw adminError("Blockchain update failed.", error);
            }
        }
        // Remove the zip file after successful extraction
        await fs_1.promises.unlink(response.path);
        return {
            message: "Update downloaded and extracted successfully",
            status: true,
        };
    }
    catch (error) {
        throw adminError("Update extraction failed.", error);
    }
}
async function fetchAllProductsUpdates() {
    try {
        const licenseConfig = (0, license_1.getLicenseConfig)();
        const apiUrl = process.env.APP_LICENSE_API_URL || licenseConfig.apiUrl;
        const response = await callApi("POST", `${apiUrl}/api/all_products_updates`, {});
        return response;
    }
    catch (error) {
        console.error("Failed to fetch all products updates:", error);
        return { status: false, message: "Failed to fetch updates", products: [] };
    }
}
const unzip = (filePath, outPath) => {
    const zip = new adm_zip_1.default(filePath);
    zip.extractAllTo(outPath, true);
};
