"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const utils_1 = require("../utils");
const utils_2 = require("./utils");
const db_1 = require("@b/db");
exports.metadata = {
    summary: "Verifies the OTP or recovery code for login",
    operationId: "verifyLoginOTP",
    tags: ["Auth"],
    description: "Verifies the OTP for login and returns a session token. If the OTP is invalid, the provided code is checked against the recovery codes.",
    requiresAuth: false,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        id: {
                            type: "string",
                            format: "uuid",
                            description: "ID of the user",
                        },
                        otp: {
                            type: "string",
                            description: "OTP or recovery code to verify",
                        },
                    },
                    required: ["id", "otp"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "OTP or recovery code verified successfully, user logged in",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: {
                                type: "string",
                                description: "Success message",
                            },
                        },
                    },
                },
            },
        },
        400: { description: "Invalid request" },
        401: { description: "Unauthorized" },
    },
};
exports.default = async (data) => {
    const { id, otp } = data.body;
    // Validate request parameters.
    (0, utils_2.validateOtpRequest)(id, otp);
    const user = await (0, utils_2.getUserWith2FA)(id);
    let secretToVerify = user.twoFactor.secret;
    let wasPlaintext = false;
    // Check and decrypt if needed
    if ((0, utils_2.isEncrypted)(secretToVerify)) {
        try {
            secretToVerify = (0, utils_2.decrypt)(secretToVerify);
        }
        catch (err) {
            throw new Error("Could not decrypt 2FA secret. User data may be corrupted.");
        }
    }
    else {
        wasPlaintext = true;
    }
    // First, attempt to verify the OTP using the authenticator.
    if (!(0, utils_2.verifyOtp)(secretToVerify, otp)) {
        // If OTP verification fails, try to consume a recovery code.
        await (0, utils_2.consumeRecoveryCode)(user.twoFactor, otp);
    }
    else if (wasPlaintext) {
        // If it worked and it was plaintext, re-save as encrypted!
        const encrypted = (0, utils_2.encrypt)(user.twoFactor.secret);
        await db_1.models.twoFactor.update({ secret: encrypted }, { where: { id: user.twoFactor.id } });
    }
    return await (0, utils_1.returnUserWithTokens)({
        user,
        message: "You have been logged in successfully",
    });
};
