"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const passwords_1 = require("@b/utils/passwords");
const db_1 = require("@b/db");
const affiliate_1 = require("@b/utils/affiliate");
const utils_1 = require("../utils");
const error_1 = require("@b/utils/error");
// Check reCAPTCHA status - use a function to check at runtime
const isRecaptchaEnabled = () => process.env.NEXT_PUBLIC_GOOGLE_RECAPTCHA_STATUS === "true";
// For backward compatibility, keep the const but use the function
const recaptchaEnabled = isRecaptchaEnabled();
// --- Helper: Sanitize Names ---
/**
 * Sanitizes user-provided names for backend storage:
 * - Removes all HTML tags
 * - Removes dangerous characters
 * - Only allows letters (unicode), spaces, hyphens, apostrophes, periods
 * - Trims and limits to 64 chars
 */
function sanitizeName(name) {
    if (typeof name !== "string")
        return "";
    // Remove HTML tags
    let sanitized = name.replace(/<.*?>/g, "");
    // Remove dangerous characters
    sanitized = sanitized.replace(/[&<>"'/\\;:]/g, "");
    // Allow only unicode letters, spaces, hyphens, apostrophes, and dots
    sanitized = sanitized.replace(/[^\p{L} \-'.]/gu, "");
    // Trim and limit length
    sanitized = sanitized.trim().slice(0, 64);
    return sanitized;
}
exports.metadata = {
    summary: "Registers a new user",
    operationId: "registerUser",
    tags: ["Auth"],
    description: "Registers a new user and returns a session token",
    requiresAuth: false,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        firstName: {
                            type: "string",
                            description: "First name of the user",
                        },
                        lastName: {
                            type: "string",
                            description: "Last name of the user",
                        },
                        email: {
                            type: "string",
                            format: "email",
                            description: "Email of the user",
                        },
                        password: {
                            type: "string",
                            description: "Password of the user",
                        },
                        ref: {
                            type: "string",
                            description: "Referral code",
                        },
                        recaptchaToken: {
                            type: "string",
                            description: "Recaptcha token if enabled",
                            nullable: true, // Always make it nullable in schema
                        },
                    },
                    required: [
                        "firstName",
                        "lastName",
                        "email",
                        "password",
                        // Don't require it in schema, validate in handler
                    ],
                },
            },
        },
    },
    responses: {
        200: {
            description: "User registered successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: {
                                type: "string",
                                description: "Success message",
                            },
                            cookies: {
                                type: "object",
                                properties: {
                                    accessToken: {
                                        type: "string",
                                        description: "Access token",
                                    },
                                    sessionId: {
                                        type: "string",
                                        description: "Session ID",
                                    },
                                    csrfToken: {
                                        type: "string",
                                        description: "CSRF token",
                                    },
                                },
                            },
                        },
                    },
                },
            },
        },
        400: {
            description: "Invalid request (e.g., email already in use)",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: {
                                type: "string",
                                description: "Error message",
                            },
                        },
                    },
                },
            },
        },
    },
};
exports.default = async (data) => {
    const { body } = data;
    let { firstName, lastName } = body;
    const { email, password, ref, recaptchaToken } = body;
    // Verify reCAPTCHA if enabled (check at runtime)
    if (isRecaptchaEnabled()) {
        if (!recaptchaToken) {
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "reCAPTCHA token is required",
            });
        }
        const isHuman = await (0, utils_1.verifyRecaptcha)(recaptchaToken);
        if (!isHuman) {
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "reCAPTCHA verification failed",
            });
        }
    }
    // --- Input Sanitization ---
    firstName = sanitizeName(firstName);
    lastName = sanitizeName(lastName);
    if (!firstName || !lastName) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Invalid name(s)" });
    }
    // Email uniqueness check
    const existingUser = await db_1.models.user.findOne({ where: { email } });
    if (existingUser && existingUser.email) {
        if (!existingUser.emailVerified &&
            process.env.NEXT_PUBLIC_VERIFY_EMAIL_STATUS === "true") {
            await (0, utils_1.sendEmailVerificationToken)(existingUser.id, existingUser.email);
            return {
                message: "User already registered but email not verified. Verification email sent.",
            };
        }
        throw (0, error_1.createError)({ statusCode: 400, message: "Email already in use" });
    }
    // Password policy check
    if (!(0, passwords_1.validatePassword)(password)) {
        throw (0, error_1.createError)({ statusCode: 400, message: "Invalid password format" });
    }
    const hashedPassword = await (0, passwords_1.hashPassword)(password);
    // Upsert roles as needed
    await db_1.models.role.upsert({ name: "User" });
    const roleName = process.env.NEXT_PUBLIC_DEMO_STATUS === "true" ? "Admin" : "User";
    await db_1.models.role.upsert({ name: roleName });
    // Fetch the role to get its ID
    const role = await db_1.models.role.findOne({ where: { name: roleName } });
    if (!role)
        throw (0, error_1.createError)({ statusCode: 500, message: "Role not found after upsert." });
    // Create the user (with sanitized names)
    const newUser = await db_1.models.user.create({
        firstName,
        lastName,
        email,
        password: hashedPassword,
        roleId: role.id,
        emailVerified: false,
    });
    if (!newUser.email) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Error creating user",
        });
    }
    // Referral code
    try {
        if (ref)
            await (0, affiliate_1.handleReferralRegister)(ref, newUser.id);
    }
    catch (error) {
        console.error("Error handling referral registration:", error);
    }
    // Email verification logic
    if (process.env.NEXT_PUBLIC_VERIFY_EMAIL_STATUS === "true") {
        await (0, utils_1.sendEmailVerificationToken)(newUser.id, newUser.email);
        return {
            message: "Registration successful, please verify your email",
        };
    }
    else {
        return await (0, utils_1.returnUserWithTokens)({
            user: newUser,
            message: "You have been registered successfully",
        });
    }
};
