"use strict";
// /server/api/auth/reset.post.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const token_1 = require("@b/utils/token");
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const emails_1 = require("@b/utils/emails");
const utils_1 = require("../utils");
// Check reCAPTCHA status - use a function to check at runtime
const isRecaptchaEnabled = () => process.env.NEXT_PUBLIC_GOOGLE_RECAPTCHA_STATUS === "true";
// For backward compatibility, keep the const but use the function
const recaptchaEnabled = isRecaptchaEnabled();
exports.metadata = {
    summary: "Initiates a password reset process for a user",
    operationId: "resetPassword",
    tags: ["Auth"],
    description: "Initiates a password reset process for a user and sends an email with a reset link",
    requiresAuth: false,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        email: {
                            type: "string",
                            format: "email",
                            description: "Email of the user",
                        },
                        recaptchaToken: {
                            type: "string",
                            description: "Recaptcha token if enabled",
                            nullable: true, // Always make it nullable in schema
                        },
                    },
                    required: [
                        "email",
                        // Don't require it in schema, validate in handler
                    ],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Password reset process initiated successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            status: {
                                type: "boolean",
                                description: "Indicates if the request was successful",
                            },
                            statusCode: {
                                type: "number",
                                description: "HTTP status code",
                                example: 200,
                            },
                            data: {
                                type: "object",
                                properties: {
                                    message: {
                                        type: "string",
                                        description: "Success message",
                                    },
                                },
                            },
                        },
                    },
                },
            },
        },
        400: {
            description: "Invalid request (e.g., missing email)",
        },
        404: {
            description: "User not found with the provided email",
        },
    },
};
exports.default = (data) => {
    const { body } = data;
    const { email, recaptchaToken } = body;
    return resetPasswordQuery(email, recaptchaToken);
};
const resetPasswordQuery = async (email, recaptchaToken) => {
    // Verify reCAPTCHA if enabled (check at runtime)
    if (isRecaptchaEnabled()) {
        if (!recaptchaToken) {
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "reCAPTCHA token is required",
            });
        }
        const isHuman = await (0, utils_1.verifyRecaptcha)(recaptchaToken);
        if (!isHuman) {
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "reCAPTCHA verification failed",
            });
        }
    }
    const user = await db_1.models.user.findOne({ where: { email } });
    if (!user) {
        throw new Error("User not found");
    }
    const resetToken = await (0, token_1.generateResetToken)({
        user: {
            id: user.id,
        },
    });
    try {
        await emails_1.emailQueue.add({
            emailData: {
                TO: user.email,
                FIRSTNAME: user.firstName,
                LAST_LOGIN: user.lastLogin,
                TOKEN: resetToken,
            },
            emailType: "PasswordReset",
        });
        return {
            message: "Email with reset instructions sent successfully",
        };
    }
    catch (error) {
        throw (0, error_1.createError)({
            message: error.message,
            statusCode: 500,
        });
    }
};
