"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const error_1 = require("@b/utils/error");
const utils_1 = require("./utils");
const query_1 = require("@b/utils/query");
exports.metadata = {
    summary: "Get price in USD for a currency",
    description: "Returns the price in USD for a given currency and wallet type.",
    operationId: "getCurrencyPriceInUSD",
    tags: ["Finance", "Currency"],
    parameters: [
        {
            name: "currency",
            in: "query",
            description: "The currency to get the price for",
            required: true,
            schema: {
                type: "string",
            },
        },
        {
            name: "type",
            in: "query",
            description: "The wallet type of the currency",
            required: true,
            schema: {
                type: "string",
            },
        },
    ],
    requiresAuth: true,
    responses: {
        200: {
            description: "Price in USD retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            ...utils_1.baseResponseSchema,
                            data: {
                                type: "number",
                                description: "Price of the currency in USD",
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Currency"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user, query } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id))
        throw (0, error_1.createError)(401, "Unauthorized");
    const { currency, type } = query;
    if (!currency || !type) {
        throw (0, error_1.createError)(400, "Missing required query parameters");
    }
    let priceUSD;
    switch (type) {
        case "FIAT":
            priceUSD = await (0, utils_1.getFiatPriceInUSD)(currency);
            break;
        case "SPOT":
            priceUSD = await (0, utils_1.getSpotPriceInUSD)(currency);
            break;
        case "ECO":
            priceUSD = await (0, utils_1.getEcoPriceInUSD)(currency);
            break;
        default:
            throw (0, error_1.createError)(400, `Invalid type: ${type}`);
    }
    return {
        status: true,
        message: "Price in USD retrieved successfully",
        data: priceUSD,
    };
};
