"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
const db_1 = require("@b/db");
const sequelize_1 = require("sequelize");
exports.metadata = {
    summary: "Verifies an Adyen payment",
    description: "Manually verifies an Adyen payment by checking the payment status and updating the transaction accordingly. This endpoint is used for manual verification when automatic webhook processing is not available.",
    operationId: "verifyAdyenPayment",
    tags: ["Finance", "Deposit"],
    requestBody: {
        description: "Payment verification data",
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        reference: {
                            type: "string",
                            description: "Transaction reference",
                        },
                        pspReference: {
                            type: "string",
                            description: "Adyen PSP reference",
                            nullable: true,
                        },
                    },
                    required: ["reference"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Payment verification completed successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            success: {
                                type: "boolean",
                                description: "Whether the verification was successful",
                            },
                            status: {
                                type: "string",
                                description: "Payment status",
                            },
                            message: {
                                type: "string",
                                description: "Verification message",
                            },
                            transaction: {
                                type: "object",
                                description: "Updated transaction details",
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Transaction"),
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    var _a, _b;
    const { user, body } = data;
    if (!user)
        throw new Error("User not authenticated");
    const { reference, pspReference } = body;
    try {
        // Find the transaction
        const transaction = await db_1.models.transaction.findOne({
            where: {
                uuid: reference,
                userId: user.id,
                type: "DEPOSIT",
                status: {
                    [sequelize_1.Op.in]: ["PENDING", "PROCESSING"],
                },
            },
        });
        if (!transaction) {
            throw new Error("Transaction not found or already processed");
        }
        // Get Adyen configuration
        const config = (0, utils_1.getAdyenConfig)();
        let paymentData;
        if (pspReference) {
            // If PSP reference is provided, query payment details directly
            try {
                const paymentDetailsRequest = {
                    pspReference,
                };
                paymentData = await (0, utils_1.makeAdyenApiRequest)("/payments/details", paymentDetailsRequest, config);
            }
            catch (error) {
                throw new Error(`Failed to fetch payment details: ${error.message}`);
            }
        }
        else {
            // If no PSP reference, we need to check if we have it in metadata
            const metadata = transaction.metadata;
            if (metadata === null || metadata === void 0 ? void 0 : metadata.pspReference) {
                const paymentDetailsRequest = {
                    pspReference: metadata.pspReference,
                };
                paymentData = await (0, utils_1.makeAdyenApiRequest)("/payments/details", paymentDetailsRequest, config);
            }
            else {
                throw new Error("PSP reference not provided and not found in transaction metadata");
            }
        }
        // Process payment result
        const resultCode = paymentData.resultCode;
        let newStatus;
        let shouldUpdateWallet = false;
        switch (resultCode) {
            case "Authorised":
                newStatus = "COMPLETED";
                shouldUpdateWallet = true;
                break;
            case "Cancelled":
            case "Error":
            case "Refused":
                newStatus = "FAILED";
                break;
            case "Pending":
            case "Received":
                newStatus = "PROCESSING";
                break;
            default:
                newStatus = "FAILED";
                break;
        }
        // Update transaction
        const updatedTransaction = await db_1.models.transaction.update({
            status: newStatus,
            metadata: {
                ...transaction.metadata,
                pspReference: paymentData.pspReference,
                resultCode,
                verifiedAt: new Date().toISOString(),
                verificationMethod: "manual",
            },
        }, {
            where: { id: transaction.id },
            returning: true,
        });
        // Update wallet if payment was successful
        if (shouldUpdateWallet) {
            const wallet = await db_1.models.wallet.findOne({
                where: {
                    userId: user.id,
                    currency: ((_a = transaction.metadata) === null || _a === void 0 ? void 0 : _a.currency) || "USD",
                    type: "FIAT",
                },
            });
            if (wallet) {
                await wallet.increment("balance", {
                    by: transaction.amount - (transaction.fee || 0),
                });
            }
            else {
                // Create wallet if it doesn't exist
                await db_1.models.wallet.create({
                    userId: user.id,
                    type: "FIAT",
                    currency: ((_b = transaction.metadata) === null || _b === void 0 ? void 0 : _b.currency) || "USD",
                    balance: transaction.amount - (transaction.fee || 0),
                });
            }
            // Create wallet transaction record
            await db_1.models.walletTransaction.create({
                walletId: wallet === null || wallet === void 0 ? void 0 : wallet.id,
                type: "DEPOSIT",
                amount: transaction.amount - (transaction.fee || 0),
                balance: wallet ? wallet.balance + (transaction.amount - (transaction.fee || 0)) : transaction.amount - (transaction.fee || 0),
                description: `Adyen deposit verification - ${reference}`,
                metadata: {
                    transactionId: transaction.id,
                    gateway: "adyen",
                    pspReference: paymentData.pspReference,
                },
            });
        }
        return {
            success: shouldUpdateWallet,
            status: newStatus,
            message: shouldUpdateWallet
                ? "Payment verified and wallet updated successfully"
                : `Payment verification completed with status: ${resultCode}`,
            transaction: updatedTransaction[1][0],
            pspReference: paymentData.pspReference,
            resultCode,
        };
    }
    catch (error) {
        console.error("Adyen payment verification error:", error);
        throw new Error(`Payment verification failed: ${error.message}`);
    }
};
