"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const query_1 = require("@b/utils/query");
const error_1 = require("@b/utils/error");
exports.metadata = {
    summary: "Performs a custom fiat deposit transaction",
    description: "Initiates a custom fiat deposit transaction for the currently authenticated user",
    operationId: "createCustomFiatDeposit",
    tags: ["Wallets"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        methodId: { type: "string", description: "Deposit method ID" },
                        amount: { type: "number", description: "Amount to deposit" },
                        currency: { type: "string", description: "Currency to deposit" },
                        customFields: {
                            type: "object",
                            description: "Custom data for the deposit",
                        },
                    },
                    required: ["methodId", "amount", "currency", "customFields"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Custom deposit transaction initiated successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            transaction: { type: "object" },
                            currency: { type: "string" },
                            method: { type: "string" },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Deposit Method"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user, body } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { methodId, amount, currency, customFields } = body;
    const userPk = await db_1.models.user.findByPk(user.id);
    if (!userPk) {
        throw (0, error_1.createError)({ statusCode: 404, message: "User not found" });
    }
    const method = await db_1.models.depositMethod.findByPk(methodId);
    if (!method) {
        throw (0, error_1.createError)({ statusCode: 404, message: "Deposit method not found" });
    }
    const currencyData = await db_1.models.currency.findOne({
        where: { id: currency },
    });
    if (!currencyData) {
        throw (0, error_1.createError)({ statusCode: 404, message: "Currency not found" });
    }
    const parsedAmount = parseFloat(amount);
    const fixedFee = method.fixedFee || 0;
    const percentageFee = method.percentageFee || 0;
    const taxAmount = parseFloat(Math.max((parsedAmount * percentageFee) / 100 + fixedFee, 0).toFixed(2));
    // Begin transaction for proper isolation
    const depositTransaction = await db_1.sequelize.transaction(async (t) => {
        // Find (and lock) the wallet row for this user/currency/type
        let wallet = await db_1.models.wallet.findOne({
            where: { userId: user.id, currency: currency, type: "FIAT" },
            transaction: t,
            lock: t.LOCK.UPDATE,
        });
        // If wallet doesn't exist, create and lock it
        if (!wallet) {
            wallet = await db_1.models.wallet.create({ userId: user.id, currency: currency, type: "FIAT" }, { transaction: t });
            wallet = await db_1.models.wallet.findOne({
                where: { id: wallet.id },
                transaction: t,
                lock: t.LOCK.UPDATE,
            });
        }
        // Record the deposit transaction
        const trx = await db_1.models.transaction.create({
            userId: user.id,
            walletId: wallet.id,
            type: "DEPOSIT",
            amount: parsedAmount,
            fee: taxAmount,
            status: "PENDING",
            metadata: JSON.stringify({
                method: method.title,
                ...customFields,
            }),
            description: `Deposit ${parsedAmount} ${wallet.currency} by ${method.title}`,
        }, { transaction: t });
        // Record admin profit if needed
        if (taxAmount > 0) {
            await db_1.models.adminProfit.create({
                amount: taxAmount,
                currency: wallet.currency,
                type: "DEPOSIT",
                transactionId: trx.id,
                description: `Admin profit from deposit fee of ${taxAmount} ${wallet.currency} by ${method.title} for user (${user.id})`,
            }, { transaction: t });
        }
        // (Optional: If you want to update balance instantly for deposits, do it here)
        // wallet.balance += parsedAmount - taxAmount;
        // await wallet.save({ transaction: t });
        return trx;
    });
    return {
        transaction: depositTransaction,
        currency,
        method: method.title,
    };
};
