"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const emails_1 = require("@b/utils/emails");
const utils_1 = require("./utils");
exports.metadata = {
    summary: 'Handles PayFast ITN webhook',
    description: 'Processes PayFast Instant Transaction Notification (ITN) callbacks',
    operationId: 'handlePayFastWebhook',
    tags: ['Finance', 'Deposit', 'PayFast', 'Webhook'],
    requiresAuth: false, // Webhooks don't use user authentication
    requestBody: {
        required: true,
        content: {
            'application/x-www-form-urlencoded': {
                schema: {
                    type: 'object',
                    properties: {
                        m_payment_id: {
                            type: 'string',
                            description: 'Merchant payment ID',
                        },
                        pf_payment_id: {
                            type: 'string',
                            description: 'PayFast payment ID',
                        },
                        payment_status: {
                            type: 'string',
                            description: 'Payment status from PayFast',
                        },
                        amount_gross: {
                            type: 'string',
                            description: 'Gross payment amount',
                        },
                        amount_fee: {
                            type: 'string',
                            description: 'PayFast processing fee',
                        },
                        amount_net: {
                            type: 'string',
                            description: 'Net amount after fees',
                        },
                        signature: {
                            type: 'string',
                            description: 'PayFast signature for verification',
                        },
                    },
                    required: ['m_payment_id', 'pf_payment_id', 'payment_status'],
                },
            },
        },
    },
    responses: {
        200: {
            description: 'Webhook processed successfully',
            content: {
                'text/plain': {
                    schema: {
                        type: 'string',
                        example: 'OK',
                    },
                },
            },
        },
        400: { description: 'Bad request - invalid webhook data' },
        500: { description: 'Internal server error' },
    },
};
exports.default = async (data) => {
    var _a, _b;
    const { body } = data;
    console.log('PayFast ITN received:', {
        m_payment_id: body.m_payment_id,
        pf_payment_id: body.pf_payment_id,
        payment_status: body.payment_status,
        amount_gross: body.amount_gross
    });
    // Validate required fields
    if (!body.m_payment_id || !body.pf_payment_id || !body.payment_status) {
        throw (0, error_1.createError)({
            statusCode: 400,
            message: 'Required webhook fields missing',
        });
    }
    // Validate configuration
    (0, utils_1.validatePayFastConfig)();
    try {
        // Validate signature
        if (utils_1.PAYFAST_CONFIG.PASSPHRASE) {
            const isValidSignature = (0, utils_1.validateSignature)(body, utils_1.PAYFAST_CONFIG.PASSPHRASE);
            if (!isValidSignature) {
                console.error('PayFast ITN signature validation failed');
                throw (0, error_1.createError)({
                    statusCode: 400,
                    message: 'Invalid webhook signature',
                });
            }
        }
        // Validate ITN with PayFast (optional additional security)
        const itnValidation = await (0, utils_1.validateITN)(body);
        if (!itnValidation.valid) {
            console.error('PayFast ITN validation failed:', itnValidation.error);
            // Log but don't fail - signature validation is primary security
        }
        // Find the transaction
        const transaction = await db_1.models.transaction.findOne({
            where: {
                uuid: body.m_payment_id
            },
            include: [
                {
                    model: db_1.models.user,
                    as: 'user',
                    attributes: ['id', 'email', 'firstName', 'lastName']
                }
            ]
        });
        if (!transaction) {
            console.error('Transaction not found for PayFast ITN:', body.m_payment_id);
            throw (0, error_1.createError)({
                statusCode: 404,
                message: 'Transaction not found',
            });
        }
        // Check if status has changed to prevent duplicate processing
        const currentStatus = transaction.status;
        const newStatus = (0, utils_1.mapPayFastStatus)(body.payment_status);
        if (currentStatus === newStatus) {
            console.log('PayFast ITN: Status unchanged, skipping processing');
            return 'OK';
        }
        // Get payment amounts
        const grossAmount = (0, utils_1.parsePayFastAmount)(body.amount_gross);
        const feeAmount = (0, utils_1.parsePayFastAmount)(body.amount_fee || '0');
        const netAmount = (0, utils_1.parsePayFastAmount)(body.amount_net || body.amount_gross) - feeAmount;
        // Start database transaction
        const dbTransaction = await db_1.sequelize.transaction();
        try {
            // Update transaction status and metadata
            await transaction.update({
                status: newStatus,
                fee: feeAmount,
                metadata: {
                    ...transaction.metadata,
                    payfast: {
                        ...(_a = transaction.metadata) === null || _a === void 0 ? void 0 : _a.payfast,
                        pf_payment_id: body.pf_payment_id,
                        payment_status: body.payment_status,
                        amount_gross: grossAmount,
                        amount_fee: feeAmount,
                        amount_net: netAmount,
                        itn_received_at: new Date().toISOString(),
                        signature_valid: true,
                        itn_valid: itnValidation.valid,
                        webhook_data: body
                    }
                }
            }, { transaction: dbTransaction });
            // If payment was successful, update user wallet
            if (newStatus === 'COMPLETED' && currentStatus !== 'COMPLETED') {
                const currency = ((_b = transaction.metadata) === null || _b === void 0 ? void 0 : _b.currency) || 'ZAR';
                // Get or create user wallet
                let wallet = await db_1.models.wallet.findOne({
                    where: {
                        userId: transaction.userId,
                        currency: currency
                    },
                    transaction: dbTransaction
                });
                if (!wallet) {
                    wallet = await db_1.models.wallet.create({
                        userId: transaction.userId,
                        currency: currency,
                        balance: 0,
                        type: 'FIAT'
                    }, { transaction: dbTransaction });
                }
                // Update wallet balance
                const newBalance = parseFloat(wallet.balance) + netAmount;
                await wallet.update({
                    balance: newBalance
                }, { transaction: dbTransaction });
                // Record admin profit from processing fees
                if (feeAmount > 0) {
                    try {
                        await db_1.models.adminProfit.create({
                            type: 'DEPOSIT_FEE',
                            amount: feeAmount,
                            currency: currency,
                            description: `PayFast processing fee for transaction ${transaction.id}`,
                            metadata: {
                                transactionId: transaction.id,
                                userId: transaction.userId,
                                gateway: 'payfast',
                                pf_payment_id: body.pf_payment_id
                            }
                        }, { transaction: dbTransaction });
                    }
                    catch (profitError) {
                        console.error('Failed to record admin profit:', profitError);
                        // Don't fail the transaction for profit recording errors
                    }
                }
                // Send confirmation email
                try {
                    await (0, emails_1.sendFiatTransactionEmail)(transaction.user, transaction, currency, newBalance);
                }
                catch (emailError) {
                    console.error('Failed to send confirmation email:', emailError);
                    // Don't fail the transaction for email errors
                }
            }
            // Commit the database transaction
            await dbTransaction.commit();
            console.log('PayFast ITN processed successfully:', {
                transactionId: transaction.id,
                oldStatus: currentStatus,
                newStatus: newStatus,
                amount: grossAmount,
                fee: feeAmount
            });
            return 'OK';
        }
        catch (dbError) {
            await dbTransaction.rollback();
            throw dbError;
        }
    }
    catch (error) {
        console.error('PayFast ITN processing error:', error);
        // Return error response but don't expose internal details
        throw (0, error_1.createError)({
            statusCode: error.statusCode || 500,
            message: 'Webhook processing failed',
        });
    }
};
