"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
exports.getKyc = getKyc;
const db_1 = require("@b/db");
const error_1 = require("@b/utils/error");
const query_1 = require("@b/utils/query");
exports.metadata = {
    summary: "Retrieves all KYC applications for the logged-in user",
    description: "Fetches all Know Your Customer (KYC) applications for the currently authenticated user. This endpoint requires user authentication and returns an array with the user’s KYC application information, including the verification status and other details.",
    operationId: "getUserKycApplications",
    tags: ["KYC"],
    responses: {
        200: {
            description: "KYC applications retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                id: { type: "number", description: "KYC ID" },
                                userId: {
                                    type: "number",
                                    description: "User ID associated with the KYC record",
                                },
                                templateId: {
                                    type: "number",
                                    description: "ID of the KYC template used",
                                },
                                data: {
                                    type: "object",
                                    description: "KYC data as a JSON object",
                                    nullable: true,
                                },
                                status: {
                                    type: "string",
                                    description: "Current status of the KYC verification",
                                    enum: ["PENDING", "APPROVED", "REJECTED"],
                                },
                                level: { type: "number", description: "Verification level" },
                                notes: {
                                    type: "string",
                                    description: "Administrative notes, if any",
                                    nullable: true,
                                },
                                createdAt: {
                                    type: "string",
                                    format: "date-time",
                                    description: "Timestamp when the KYC record was created",
                                },
                                updatedAt: {
                                    type: "string",
                                    format: "date-time",
                                    description: "Timestamp when the KYC record was last updated",
                                },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Kyc"),
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    var _a;
    if (!((_a = data.user) === null || _a === void 0 ? void 0 : _a.id))
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    return getKyc(data.user.id);
};
async function getKyc(userId) {
    const responses = await db_1.models.kycApplication.findAll({
        where: {
            userId,
        },
        include: [
            {
                model: db_1.models.kycLevel,
                as: "level",
                paranoid: false, // kycLevel doesn't have soft deletes
            },
            {
                model: db_1.models.kycVerificationResult,
                as: "verificationResult",
            },
        ],
    });
    if (responses.length === 0) {
        throw new Error("KYC records not found");
    }
    return responses.map((response) => response.get({ plain: true }));
}
