"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateUserQuery = exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const db_1 = require("@b/db");
const promises_1 = require("fs/promises");
exports.metadata = {
    summary: "Updates the profile of the current user",
    description: "Updates the profile of the currently authenticated user",
    operationId: "updateProfile",
    tags: ["Auth"],
    requiresAuth: true,
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        firstName: {
                            type: "string",
                            description: "First name of the user",
                        },
                        lastName: {
                            type: "string",
                            description: "Last name of the user",
                        },
                        metadata: {
                            type: "object",
                            description: "Metadata of the user",
                        },
                        avatar: {
                            type: "string",
                            description: "Avatar of the user",
                            nullable: true,
                        },
                        phone: {
                            type: "string",
                            description: "Phone number of the user",
                        },
                        twoFactor: {
                            type: "boolean",
                            description: "Two-factor authentication status",
                        },
                        profile: {
                            type: "object",
                            nullable: true,
                            properties: {
                                bio: {
                                    type: "string",
                                    description: "User bio",
                                    nullable: true,
                                },
                                location: {
                                    type: "object",
                                    nullable: true,
                                    properties: {
                                        address: {
                                            type: "string",
                                            description: "User address",
                                            nullable: true,
                                        },
                                        city: {
                                            type: "string",
                                            description: "User city",
                                            nullable: true,
                                        },
                                        country: {
                                            type: "string",
                                            description: "User country",
                                            nullable: true,
                                        },
                                        zip: {
                                            type: "string",
                                            description: "User zip code",
                                            nullable: true,
                                        },
                                    },
                                },
                                social: {
                                    type: "object",
                                    nullable: true,
                                    properties: {
                                        twitter: {
                                            type: "string",
                                            description: "Twitter profile",
                                            nullable: true,
                                        },
                                        dribbble: {
                                            type: "string",
                                            description: "Dribbble profile",
                                            nullable: true,
                                        },
                                        instagram: {
                                            type: "string",
                                            description: "Instagram profile",
                                            nullable: true,
                                        },
                                        github: {
                                            type: "string",
                                            description: "GitHub profile",
                                            nullable: true,
                                        },
                                        gitlab: {
                                            type: "string",
                                            description: "GitLab profile",
                                            nullable: true,
                                        },
                                        telegram: {
                                            type: "string",
                                            description: "Telegram username",
                                            nullable: true,
                                        },
                                    },
                                },
                            },
                        },
                        settings: {
                            type: "object",
                            description: "Notification settings for the user",
                            properties: {
                                email: {
                                    type: "boolean",
                                    description: "Email notifications enabled or disabled",
                                },
                                sms: {
                                    type: "boolean",
                                    description: "SMS notifications enabled or disabled",
                                },
                                push: {
                                    type: "boolean",
                                    description: "Push notifications enabled or disabled",
                                },
                            },
                        },
                    },
                },
            },
        },
    },
    responses: {
        200: {
            description: "User profile updated successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: {
                                type: "string",
                                description: "Success message",
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("User"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    var _a;
    const { user, body } = data;
    if (!user) {
        throw new Error("Authentication required to update profile");
    }
    const { firstName, lastName, metadata, avatar, phone, twoFactor, profile, settings, } = body;
    return await (0, exports.updateUserQuery)(user.id, firstName, lastName, metadata, avatar, phone, twoFactor, profile, settings, (_a = user.avatar) !== null && _a !== void 0 ? _a : undefined // Passing the original avatar path to check for unlinking
    );
};
const updateUserQuery = async (id, firstName, lastName, metadata, avatar, phone, twoFactor, profile, settings, originalAvatar // Original avatar path
) => {
    // Prepare updateData with only the allowed fields
    const updateData = {};
    if (firstName !== undefined)
        updateData.firstName = firstName;
    if (lastName !== undefined)
        updateData.lastName = lastName;
    if (metadata !== undefined)
        updateData.metadata = metadata;
    if (avatar !== undefined)
        updateData.avatar = avatar;
    if (phone !== undefined)
        updateData.phone = phone;
    if (twoFactor !== undefined)
        updateData.twoFactor = twoFactor;
    if (profile !== undefined)
        updateData.profile = profile;
    if (settings !== undefined) {
        updateData.settings =
            typeof settings === "string" ? JSON.parse(settings) : settings;
    }
    // Handle avatar removal if necessary
    if (avatar === null && originalAvatar) {
        try {
            await (0, promises_1.unlink)(originalAvatar);
        }
        catch (error) {
            console.error(`Failed to unlink avatar: ${error}`);
            throw new Error("Failed to unlink avatar from server");
        }
    }
    // Perform the update
    await db_1.models.user.update(updateData, {
        where: { id },
    });
    return { message: "Profile updated successfully" };
};
exports.updateUserQuery = updateUserQuery;
