"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.models = exports.sequelize = exports.db = exports.SequelizeSingleton = void 0;
const sequelize_1 = require("sequelize");
const init_1 = require("../models/init");
const worker_threads_1 = require("worker_threads");
class SequelizeSingleton {
    constructor() {
        // Log database configuration for debugging
        console.log(`\x1b[36mDatabase Configuration:\x1b[0m`);
        console.log(`  DB_NAME: ${process.env.DB_NAME || '(not set)'}`);
        console.log(`  DB_USER: ${process.env.DB_USER || '(not set)'}`);
        console.log(`  DB_PASSWORD: ${process.env.DB_PASSWORD !== undefined ? (process.env.DB_PASSWORD === '' ? '(empty)' : '[HIDDEN]') : '(not set)'}`);
        console.log(`  DB_HOST: ${process.env.DB_HOST || '(not set)'}`);
        console.log(`  DB_PORT: ${process.env.DB_PORT || '(not set)'}`);
        if (!process.env.DB_NAME || !process.env.DB_USER || !process.env.DB_HOST) {
            throw new Error('Missing required database environment variables. Please check your .env file.');
        }
        this.sequelize = new sequelize_1.Sequelize(process.env.DB_NAME, process.env.DB_USER, process.env.DB_PASSWORD || '', // Use empty string if undefined
        {
            host: process.env.DB_HOST,
            dialect: "mysql",
            port: Number(process.env.DB_PORT),
            logging: false,
            dialectOptions: {
                charset: "utf8mb4",
            },
            define: {
                charset: "utf8mb4",
                collate: "utf8mb4_unicode_ci",
            },
        });
        if (!this.sequelize) {
            throw new Error("Failed to create Sequelize instance");
        }
        this.models = this.initModels();
    }
    static getInstance() {
        if (!SequelizeSingleton.instance) {
            SequelizeSingleton.instance = new SequelizeSingleton();
        }
        return SequelizeSingleton.instance;
    }
    async initialize() {
        if (worker_threads_1.isMainThread) {
            await this.syncDatabase();
            console.log(`\x1b[36mMain Thread: Database synced successfully...\x1b[0m`);
        }
    }
    getSequelize() {
        return this.sequelize;
    }
    initModels() {
        const models = (0, init_1.initModels)(this.sequelize);
        return models;
    }
    async syncDatabase() {
        try {
            await this.sequelize.sync({ alter: true });
        }
        catch (error) {
            console.error("Database sync failed:", error);
            throw error;
        }
    }
}
exports.SequelizeSingleton = SequelizeSingleton;
exports.db = SequelizeSingleton.getInstance();
exports.sequelize = exports.db.getSequelize();
exports.models = exports.db.models;
exports.default = exports.db;
