"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.routeCache = void 0;
exports.setupApiRoutes = setupApiRoutes;
const promises_1 = __importDefault(require("fs/promises"));
const path_1 = __importDefault(require("path"));
const Middleware_1 = require("../handler/Middleware");
const validation_1 = require("@b/utils/validation");
const constants_1 = require("@b/utils/constants");
const logger_1 = require("@b/utils/logger");
const Websocket_1 = require("./Websocket");
// Use .js extension in production, otherwise .ts for development.
const fileExtension = constants_1.isProduction ? ".js" : ".ts";
// A typed cache for routes to avoid re-importing modules.
exports.routeCache = new Map();
/**
 * Recursively sets up API routes from a directory structure.
 * - Processes directories and files.
 * - Skips certain folders/files.
 * - Supports dynamic route parameters via bracket syntax.
 *
 * @param app - The application instance (e.g. an Express-like router).
 * @param startPath - The directory path where routes are defined.
 * @param basePath - The API base path (default is "/api").
 */
async function setupApiRoutes(app, startPath, basePath = "/api") {
    try {
        const entries = await promises_1.default.readdir(startPath, { withFileTypes: true });
        // Sort so that file entries come before directories and bracketed directories come last.
        const sortedEntries = entries.sort((a, b) => {
            if (a.isDirectory() && !b.isDirectory())
                return 1;
            if (!a.isDirectory() && b.isDirectory())
                return -1;
            if (a.isDirectory() && b.isDirectory()) {
                const aHasBrackets = a.name.includes("[");
                const bHasBrackets = b.name.includes("[");
                if (aHasBrackets && !bHasBrackets)
                    return 1;
                if (!aHasBrackets && bHasBrackets)
                    return -1;
            }
            return 0;
        });
        for (const entry of sortedEntries) {
            const entryPath = (0, validation_1.sanitizePath)(path_1.default.join(startPath, entry.name));
            // Skip certain directories and files.
            if ((entry.isDirectory() && entry.name === "util") ||
                entry.name === `queries${fileExtension}` ||
                entry.name === `utils${fileExtension}`) {
                continue;
            }
            // If entry is a directory, update the basePath and recurse.
            if (entry.isDirectory()) {
                let newBasePath = basePath;
                // If the folder name is wrapped in parentheses (grouping folder), skip it.
                if (!/^\(.*\)$/.test(entry.name)) {
                    newBasePath = `${basePath}/${entry.name.replace(/\[(\w+)\]/, ":$1")}`;
                }
                await setupApiRoutes(app, entryPath, newBasePath);
                continue;
            }
            // For files: determine the route path and HTTP method.
            const [fileName, method] = entry.name.split(".");
            let routePath = basePath + (fileName !== "index" ? `/${fileName}` : "");
            // Convert bracketed parameters (e.g. [id]) to Express-like ":id" syntax.
            routePath = routePath
                .replace(/\[(\w+)\]/g, ":$1")
                .replace(/\.get|\.post|\.put|\.delete|\.del|\.ws/, "");
            // Register the route if the HTTP method exists on the app.
            if (typeof app[method] === "function") {
                if (method === "ws") {
                    (0, Websocket_1.setupWebSocketEndpoint)(app, routePath, entryPath);
                }
                else {
                    await handleHttpMethod(app, method, routePath, entryPath);
                }
            }
        }
    }
    catch (error) {
        (0, logger_1.logError)("setupApiRoutes", error, startPath);
        throw error;
    }
}
/**
 * Registers an HTTP route.
 *
 * It caches the route module (handler and metadata), parses the request body,
 * and then runs through a middleware chain (including API verification, rate limiting,
 * authentication, and role/maintenance checks) before handling the request.
 *
 * @param app - The application instance.
 * @param method - The HTTP method (e.g. "get", "post").
 * @param routePath - The full route path.
 * @param entryPath - The file system path for the route handler.
 */
async function handleHttpMethod(app, method, routePath, entryPath) {
    app[method](routePath, async (res, req) => {
        const startTime = Date.now();
        let metadata, handler;
        const cached = exports.routeCache.get(entryPath);
        if (cached) {
            handler = cached.handler;
            metadata = cached.metadata;
            req.setMetadata(metadata);
        }
        else {
            try {
                const handlerModule = await Promise.resolve(`${entryPath}`).then(s => __importStar(require(s)));
                handler = handlerModule.default;
                if (!handler) {
                    throw new Error(`Handler not found for ${entryPath}`);
                }
                metadata = handlerModule.metadata;
                if (!metadata) {
                    throw new Error(`Metadata not found for ${entryPath}`);
                }
                req.setMetadata(metadata);
                exports.routeCache.set(entryPath, { handler, metadata });
            }
            catch (error) {
                (0, logger_1.logError)("route", error, entryPath);
                res.handleError(500, error.message);
                return;
            }
        }
        if (typeof handler !== "function") {
            throw new Error(`Handler is not a function for ${entryPath}`);
        }
        try {
            await req.parseBody();
        }
        catch (error) {
            (0, logger_1.logError)("route", error, entryPath);
            res.handleError(400, `Invalid request body: ${error.message}`);
            return;
        }
        // Benchmark the request and log performance with color-coded labels.
        const endBenchmarking = () => {
            const duration = Date.now() - startTime;
            let color = "\x1b[0m";
            let label = "FAST";
            if (duration > 1000) {
                color = "\x1b[41m";
                label = "VERY SLOW";
            }
            else if (duration > 500) {
                color = "\x1b[31m";
                label = "SLOW";
            }
            else if (duration > 200) {
                color = "\x1b[33m";
                label = "MODERATE";
            }
            else if (duration > 100) {
                color = "\x1b[32m";
                label = "GOOD";
            }
            else if (duration > 50) {
                color = "\x1b[36m";
                label = "FAST";
            }
            else if (duration > 10) {
                color = "\x1b[34m";
                label = "VERY FAST";
            }
            else {
                color = "\x1b[35m";
                label = "EXCELLENT";
            }
            console.log(`${color}[${label}] Request to ${routePath} (${method.toUpperCase()}) - Duration: ${duration}ms\x1b[0m`);
        };
        // Determine the middleware chain based on metadata flags.
        if (metadata.requiresApi) {
            await (0, Middleware_1.handleApiVerification)(res, req, async () => {
                await handleRequest(res, req, handler, entryPath);
                endBenchmarking();
            });
            return;
        }
        if (!metadata.requiresAuth) {
            await handleRequest(res, req, handler, entryPath);
            endBenchmarking();
            return;
        }
        await (0, Middleware_1.rateLimit)(res, req, async () => {
            await (0, Middleware_1.authenticate)(res, req, async () => {
                await (0, Middleware_1.rolesGate)(app, res, req, routePath, method, async () => {
                    await (0, Middleware_1.siteMaintenanceAccessGate)(app, res, req, async () => {
                        await handleRequest(res, req, handler, entryPath);
                        endBenchmarking();
                    });
                });
            });
        });
    });
}
/**
 * Executes the route handler and sends the response.
 *
 * If an error occurs, it logs the error and sends an error response.
 *
 * @param res - The response object.
 * @param req - The request object.
 * @param handler - The route handler function.
 * @param entryPath - The file system path for logging errors.
 */
async function handleRequest(res, req, handler, entryPath) {
    try {
        const result = await handler(req);
        res.sendResponse(req, 200, result);
    }
    catch (error) {
        (0, logger_1.logError)("route", error, entryPath);
        const statusCode = error.statusCode || 500;
        const message = error.message || "Internal Server Error";
        // Handle validation errors by sending a custom response
        if (error.validationErrors) {
            res.sendResponse(req, statusCode, {
                message,
                statusCode,
                validationErrors: error.validationErrors,
            });
            return;
        }
        res.handleError(statusCode, message);
    }
}
