"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createWorker = void 0;
exports.fetchFiatCurrencyPrices = fetchFiatCurrencyPrices;
exports.cacheExchangeCurrencies = cacheExchangeCurrencies;
exports.processCurrenciesPrices = processCurrenciesPrices;
exports.updateCurrencyPricesBulk = updateCurrencyPricesBulk;
const bullmq_1 = require("bullmq");
const db_1 = require("@b/db");
const cache_1 = require("@b/utils/cache");
const utils_1 = require("@b/api/finance/currency/utils");
const index_get_1 = require("@b/api/exchange/currency/index.get");
const exchange_1 = __importDefault(require("@b/utils/exchange"));
const redis_1 = require("@b/utils/redis");
const logger_1 = require("./logger");
const utils_2 = require("@b/api/exchange/utils");
const wallet_1 = require("./crons/wallet");
const forex_1 = require("./crons/forex");
const ico_1 = require("./crons/ico");
const staking_1 = require("./crons/staking");
const mailwizard_1 = require("./crons/mailwizard");
const investment_1 = require("./crons/investment");
const aiInvestment_1 = require("./crons/aiInvestment");
const order_1 = require("./crons/order");
const userBlock_1 = require("./crons/userBlock");
// Safe import for ecosystem cron functions
async function processPendingEcoWithdrawals() {
    try {
        // @ts-ignore - Dynamic import for optional extension
        const module = await Promise.resolve().then(() => __importStar(require("../api/(ext)/ecosystem/utils/cron")));
        return module.processPendingEcoWithdrawals();
    }
    catch (error) {
        console.log("Ecosystem cron extension not available, skipping eco withdrawals processing");
    }
}
const broadcast_1 = require("./crons/broadcast");
const btc_deposit_scanner_1 = __importDefault(require("./crons/btc-deposit-scanner"));
const redis = redis_1.RedisSingleton.getInstance();
class CronJobManager {
    constructor() {
        this.cronJobs = [];
        this.loadNormalCronJobs();
    }
    static async getInstance() {
        if (!CronJobManager.instance) {
            CronJobManager.instance = new CronJobManager();
            await CronJobManager.instance.loadAddonCronJobs();
        }
        return CronJobManager.instance;
    }
    loadNormalCronJobs() {
        this.cronJobs.push({
            name: "processGeneralInvestments",
            title: "Process General Investments",
            period: 60 * 60 * 1000,
            description: "Processes active General investments.",
            function: "processGeneralInvestments",
            handler: investment_1.processGeneralInvestments,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processPendingOrders",
            title: "Process Pending Orders",
            period: 60 * 60 * 1000,
            description: "Processes pending binary orders.",
            function: "processPendingOrders",
            handler: order_1.processPendingOrders,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "fetchFiatCurrencyPrices",
            title: "Fetch Fiat Currency Prices",
            period: 30 * 60 * 1000,
            description: "Fetches the latest fiat currency prices.",
            function: "fetchFiatCurrencyPrices",
            handler: fetchFiatCurrencyPrices,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processCurrenciesPrices",
            title: "Process Currencies Prices",
            period: 2 * 60 * 1000,
            description: "Updates the prices of all exchange currencies in the database.",
            function: "processCurrenciesPrices",
            handler: processCurrenciesPrices,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processSpotPendingDeposits",
            title: "Process Pending Spot Deposits",
            period: 15 * 60 * 1000,
            description: "Processes pending spot wallet deposits.",
            function: "processSpotPendingDeposits",
            handler: wallet_1.processSpotPendingDeposits,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processPendingWithdrawals",
            title: "Process Pending Withdrawals",
            period: 30 * 60 * 1000,
            description: "Processes pending spot wallet withdrawals.",
            function: "processPendingWithdrawals",
            handler: wallet_1.processPendingWithdrawals,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processWalletPnl",
            title: "Process Wallet PnL",
            period: 24 * 60 * 60 * 1000,
            description: "Processes wallet PnL for all users.",
            function: "processWalletPnl",
            handler: wallet_1.processWalletPnl,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "cleanupOldPnlRecords",
            title: "Cleanup Old PnL Records",
            period: 24 * 60 * 60 * 1000,
            description: "Removes old PnL records and zero balance records.",
            function: "cleanupOldPnlRecords",
            handler: wallet_1.cleanupOldPnlRecords,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "processExpiredUserBlocks",
            title: "Process Expired User Blocks",
            period: 15 * 60 * 1000, // Run every 15 minutes
            description: "Automatically unblocks users whose temporary blocks have expired.",
            function: "processExpiredUserBlocks",
            handler: userBlock_1.processExpiredUserBlocks,
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        }, {
            name: "btcDepositScanner",
            title: "Bitcoin Deposit Scanner",
            period: 60 * 1000, // Run every 60 seconds
            description: "Scans all BTC wallets for deposits using Bitcoin Core node (only when BTC_NODE=node).",
            function: "btcDepositScanner",
            handler: async () => {
                const scanner = btc_deposit_scanner_1.default.getInstance();
                await scanner.start();
            },
            lastRun: null,
            lastRunError: null,
            category: "normal",
            status: "idle",
            progress: 0,
            lastExecutions: [],
            nextScheduledRun: null,
        });
    }
    async loadAddonCronJobs() {
        const addonCronJobs = {
            ecosystem: [
                {
                    name: "processPendingEcoWithdrawals",
                    title: "Process Pending Ecosystem Withdrawals",
                    period: 30 * 60 * 1000,
                    description: "Processes pending funding wallet withdrawals.",
                    function: "processPendingEcoWithdrawals",
                    handler: processPendingEcoWithdrawals,
                    lastRun: null,
                    lastRunError: null,
                    category: "ecosystem",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
            ai_investment: [
                {
                    name: "processAiInvestments",
                    title: "Process AI Investments",
                    period: 60 * 60 * 1000,
                    description: "Processes active AI investments.",
                    function: "processAiInvestments",
                    handler: aiInvestment_1.processAiInvestments,
                    lastRun: null,
                    lastRunError: null,
                    category: "ai_investment",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
            forex: [
                {
                    name: "processForexInvestments",
                    title: "Process Forex Investments",
                    period: 60 * 60 * 1000,
                    description: "Processes active Forex investments.",
                    function: "processForexInvestments",
                    handler: forex_1.processForexInvestments,
                    lastRun: null,
                    lastRunError: null,
                    category: "forex",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
            ico: [
                {
                    name: "processIcoOfferings",
                    title: "Process ICO Phases",
                    period: 60 * 60 * 1000,
                    description: "Processes ICO offerings and updates their status.",
                    function: "processIcoOfferings",
                    handler: ico_1.processIcoOfferings,
                    lastRun: null,
                    lastRunError: null,
                    category: "ico",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
            staking: [
                {
                    name: "processStakingPositions",
                    title: "Process Staking Logs",
                    period: 60 * 60 * 1000,
                    description: "Processes staking positions and rewards users accordingly.",
                    function: "processStakingPositions",
                    handler: staking_1.processStakingPositions,
                    lastRun: null,
                    lastRunError: null,
                    category: "staking",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
            mailwizard: [
                {
                    name: "processMailwizardCampaigns",
                    title: "Process Mailwizard Campaigns",
                    period: 60 * 60 * 1000,
                    description: "Processes Mailwizard campaigns and sends emails.",
                    function: "processMailwizardCampaigns",
                    handler: mailwizard_1.processMailwizardCampaigns,
                    lastRun: null,
                    lastRunError: null,
                    category: "mailwizard",
                    status: "idle",
                    progress: 0,
                    lastExecutions: [],
                    nextScheduledRun: null,
                },
            ],
        };
        const cacheManager = cache_1.CacheManager.getInstance();
        const extensions = await cacheManager.getExtensions();
        for (const addon of Object.keys(addonCronJobs)) {
            if (extensions.has(addon)) {
                addonCronJobs[addon].forEach((cronJob) => {
                    if (!this.isCronJobPresent(this.cronJobs, cronJob.name)) {
                        this.cronJobs.push(cronJob);
                    }
                });
            }
        }
    }
    getCronJobs() {
        return this.cronJobs;
    }
    // Updated to also record execution time and next scheduled run
    updateJobStatus(name, lastRun, lastRunError, executionTime, nextScheduledRun) {
        const job = this.cronJobs.find((job) => job.name === name);
        if (job) {
            job.lastRun = lastRun;
            job.lastRunError = lastRunError;
            // Update job status based on execution result
            if (lastRunError) {
                job.status = "failed";
            }
            else {
                job.status = "completed";
            }
            if (executionTime !== undefined) {
                job.executionTime = executionTime;
            }
            if (nextScheduledRun) {
                job.nextScheduledRun = nextScheduledRun;
            }
            // Add execution to historical metrics
            if (!job.lastExecutions) {
                job.lastExecutions = [];
            }
            job.lastExecutions.unshift({
                timestamp: lastRun,
                duration: executionTime || 0,
                status: lastRunError ? "failed" : "completed"
            });
            // Keep only last 10 executions for memory efficiency
            if (job.lastExecutions.length > 10) {
                job.lastExecutions = job.lastExecutions.slice(0, 10);
            }
            // Calculate success rate
            const totalExecutions = job.lastExecutions.length;
            const successfulExecutions = job.lastExecutions.filter(exec => exec.status === "completed").length;
            job.successRate = totalExecutions > 0 ? Math.round((successfulExecutions / totalExecutions) * 100) : 0;
            // Set job back to idle after a short delay (to allow UI to show completed status)
            setTimeout(() => {
                if (job.status === "completed" || job.status === "failed") {
                    job.status = "idle";
                    job.progress = 0;
                }
            }, 5000); // Reset to idle after 5 seconds
        }
    }
    // New method to update job status during execution
    updateJobRunningStatus(name, status, progress) {
        const job = this.cronJobs.find((job) => job.name === name);
        if (job) {
            job.status = status;
            if (progress !== undefined) {
                job.progress = progress;
            }
            // Reset progress when job completes or fails
            if (status === "completed" || status === "failed") {
                job.progress = 0;
            }
        }
    }
    // Method to manually trigger a cron job (for testing purposes)
    async triggerJob(name) {
        const job = this.cronJobs.find((job) => job.name === name);
        if (!job) {
            return false;
        }
        // Don't trigger if job is already running
        if (job.status === "running") {
            return false;
        }
        const startTime = Date.now();
        try {
            this.updateJobRunningStatus(name, "running", 0);
            await job.handler();
            const executionTime = Date.now() - startTime;
            this.updateJobStatus(name, new Date(startTime), null, executionTime);
            return true;
        }
        catch (error) {
            const executionTime = Date.now() - startTime;
            this.updateJobStatus(name, new Date(startTime), error.message, executionTime);
            (0, logger_1.logError)("manual trigger", error, __filename);
            return false;
        }
    }
    isCronJobPresent(cronJobs, jobName) {
        return cronJobs.some((job) => job.name === jobName);
    }
}
const createWorker = async (name, handler, period, concurrency = 1 // default concurrency set to 1; adjust as needed
) => {
    const cronJobManager = await CronJobManager.getInstance();
    try {
        const queue = new bullmq_1.Queue(name, {
            connection: {
                host: "127.0.0.1",
                port: 6379,
            },
        });
        // Test Redis connection before creating worker
        await queue.waitUntilReady();
        // Worker with added concurrency option
        const worker = new bullmq_1.Worker(name, async (job) => {
            const startTime = Date.now();
            try {
                // Set job status to running when it starts
                cronJobManager.updateJobRunningStatus(name, "running", 0);
                // Broadcast status change to WebSocket clients
                (0, broadcast_1.broadcastStatus)(name, "running");
                await handler();
                const executionTime = Date.now() - startTime;
                const nextScheduledRun = new Date(Date.now() + period);
                cronJobManager.updateJobStatus(name, new Date(startTime), null, executionTime, nextScheduledRun);
                // Broadcast completion status
                (0, broadcast_1.broadcastStatus)(name, "completed", { duration: executionTime });
            }
            catch (error) {
                const executionTime = Date.now() - startTime;
                const nextScheduledRun = new Date(Date.now() + period);
                cronJobManager.updateJobStatus(name, new Date(startTime), error.message, executionTime, nextScheduledRun);
                // Broadcast failure status
                (0, broadcast_1.broadcastStatus)(name, "failed");
                (0, logger_1.logError)("worker", error, __filename);
                throw error;
            }
        }, {
            connection: {
                host: "127.0.0.1",
                port: 6379,
            },
            concurrency, // worker concurrency
        });
        // Listen for worker errors
        worker.on('error', (error) => {
            console.error(`\x1b[31mWorker ${name} error: ${error.message}\x1b[0m`);
            (0, logger_1.logError)(`worker-${name}`, error, __filename);
        });
        worker.on('failed', (job, error) => {
            console.error(`\x1b[31mJob ${name} failed: ${error.message}\x1b[0m`);
        });
        // Use a deterministic jobId to prevent duplicate scheduling and add a backoff strategy for retries.
        await queue.add(name, {}, {
            jobId: `repeatable-${name}`,
            repeat: { every: period, startDate: new Date(Date.now() - period) },
            backoff: { type: "exponential", delay: Math.floor(period / 2) },
        });
        console.log(`\x1b[32mCron worker ${name} successfully scheduled\x1b[0m`);
    }
    catch (error) {
        console.error(`\x1b[31mFailed to create cron worker ${name}: ${error.message}\x1b[0m`);
        console.error(`\x1b[33mMake sure Redis is running on 127.0.0.1:6379\x1b[0m`);
        (0, logger_1.logError)(`createWorker-${name}`, error, __filename);
        throw error;
    }
};
exports.createWorker = createWorker;
async function fetchFiatCurrencyPrices() {
    const cronName = "fetchFiatCurrencyPrices";
    const startTime = Date.now();
    (0, broadcast_1.broadcastStatus)(cronName, "running");
    (0, broadcast_1.broadcastLog)(cronName, "Starting fetch fiat currency prices");
    const baseCurrency = "USD";
    const provider = process.env.APP_FIAT_RATES_PROVIDER || "openexchangerates";
    (0, broadcast_1.broadcastLog)(cronName, `Using provider: ${provider}, baseCurrency: ${baseCurrency}`);
    try {
        switch (provider.toLowerCase()) {
            case "openexchangerates":
                (0, broadcast_1.broadcastLog)(cronName, "Fetching rates from OpenExchangeRates");
                await fetchOpenExchangeRates(baseCurrency);
                break;
            case "exchangerate-api":
                (0, broadcast_1.broadcastLog)(cronName, "Fetching rates from ExchangeRate API");
                await fetchExchangeRateApi(baseCurrency);
                break;
            default:
                throw new Error(`Unsupported fiat rates provider: ${provider}`);
        }
        (0, broadcast_1.broadcastStatus)(cronName, "completed", {
            duration: Date.now() - startTime,
        });
        (0, broadcast_1.broadcastLog)(cronName, "Fetch fiat currency prices completed", "success");
    }
    catch (error) {
        (0, logger_1.logError)("fetchFiatCurrencyPrices", error, __filename);
        (0, broadcast_1.broadcastStatus)(cronName, "failed");
        (0, broadcast_1.broadcastLog)(cronName, `Fetch fiat currency prices failed: ${error.message}`, "error");
        // Don't throw - allow other operations to continue
        console.error(`[CRON_ERROR] Fiat currency prices update failed, but continuing normal operations`);
    }
}
async function fetchOpenExchangeRates(baseCurrency) {
    const cronName = "fetchOpenExchangeRates";
    (0, broadcast_1.broadcastLog)(cronName, `Starting OpenExchangeRates API call with baseCurrency: ${baseCurrency}`);
    const openExchangeRatesApiKey = process.env.APP_OPENEXCHANGERATES_APP_ID;
    const openExchangeRatesUrl = `https://openexchangerates.org/api/latest.json?appId=${openExchangeRatesApiKey}&base=${baseCurrency}`;
    const frankfurterApiUrl = `https://api.frankfurter.app/latest?from=${baseCurrency}`;
    try {
        const data = await fetchWithTimeout(openExchangeRatesUrl, 30000); // Increase timeout to 30 seconds
        (0, broadcast_1.broadcastLog)(cronName, "Data fetched from OpenExchangeRates API");
        if (data && data.rates) {
            await updateRatesFromData(data.rates);
            (0, broadcast_1.broadcastLog)(cronName, "Rates updated from OpenExchangeRates data", "success");
        }
        else {
            throw new Error("Invalid data format received from OpenExchangeRates API");
        }
    }
    catch (error) {
        (0, logger_1.logError)("fetchOpenExchangeRates - OpenExchangeRates", error, __filename);
        (0, broadcast_1.broadcastLog)(cronName, `OpenExchangeRates API failed: ${error.message}`, "error");
        (0, broadcast_1.broadcastLog)(cronName, "Attempting fallback with Frankfurter API");
        try {
            const data = await fetchWithTimeout(frankfurterApiUrl, 30000); // Increase timeout to 30 seconds
            (0, broadcast_1.broadcastLog)(cronName, "Data fetched from Frankfurter API");
            if (data && data.rates) {
                await updateRatesFromData(data.rates);
                (0, broadcast_1.broadcastLog)(cronName, "Rates updated from Frankfurter API data", "success");
            }
            else {
                throw new Error("Invalid data format received from Frankfurter API");
            }
        }
        catch (fallbackError) {
            (0, logger_1.logError)("fetchOpenExchangeRates - Frankfurter", fallbackError, __filename);
            (0, broadcast_1.broadcastLog)(cronName, `Fallback Frankfurter API failed: ${fallbackError.message}`, "error");
            // Log error but don't throw - allow operations to continue with cached rates
            console.error(`[CRON_ERROR] Both fiat API calls failed: ${error.message}, ${fallbackError.message}`);
            return; // Exit gracefully
        }
    }
}
async function fetchExchangeRateApi(baseCurrency) {
    const cronName = "fetchExchangeRateApi";
    (0, broadcast_1.broadcastLog)(cronName, `Starting ExchangeRate API call with baseCurrency: ${baseCurrency}`);
    const exchangeRateApiKey = process.env.APP_EXCHANGERATE_API_KEY;
    if (!exchangeRateApiKey) {
        throw new Error("APP_EXCHANGERATE_API_KEY is not configured in environment variables");
    }
    const exchangeRateApiUrl = `https://v6.exchangerate-api.com/v6/${exchangeRateApiKey}/latest/${baseCurrency}`;
    try {
        const data = await fetchWithTimeout(exchangeRateApiUrl, 30000); // Increase timeout to 30 seconds
        (0, broadcast_1.broadcastLog)(cronName, "Data fetched from ExchangeRate API");
        if (data && data.conversion_rates) {
            await updateRatesFromData(data.conversion_rates);
            (0, broadcast_1.broadcastLog)(cronName, "Rates updated from ExchangeRate API data", "success");
        }
        else {
            throw new Error("Invalid data format received from ExchangeRate API");
        }
    }
    catch (error) {
        (0, logger_1.logError)("fetchExchangeRateApi", error, __filename);
        (0, broadcast_1.broadcastLog)(cronName, `ExchangeRate API call failed: ${error.message}`, "error");
        // Don't throw - allow operations to continue with cached rates
        console.error(`[CRON_ERROR] ExchangeRate API failed: ${error.message}`);
        return; // Exit gracefully
    }
}
async function fetchWithTimeout(url, timeout = 5000) {
    // Note: This helper function does not use broadcast logging since it's a low-level utility.
    const controller = new AbortController();
    const id = setTimeout(() => controller.abort(), timeout);
    try {
        const response = await fetch(url, { signal: controller.signal });
        if (!response.ok) {
            switch (response.status) {
                case 401:
                    throw new Error("Unauthorized: Invalid API key.");
                case 403:
                    throw new Error("Forbidden: Access denied.");
                case 429:
                    throw new Error("Too Many Requests: Rate limit exceeded.");
                case 500:
                    throw new Error("Internal Server Error: The API is currently unavailable.");
                default:
                    throw new Error(`Network response was not ok: ${response.statusText}`);
            }
        }
        const data = await response.json();
        return data;
    }
    finally {
        clearTimeout(id);
    }
}
async function updateRatesFromData(exchangeRates) {
    const cronName = "updateRatesFromData";
    (0, broadcast_1.broadcastLog)(cronName, "Starting update of currency rates from fetched data");
    const ratesToUpdate = {};
    const currenciesRaw = await redis.get("currencies");
    let currencies;
    if (!currenciesRaw) {
        (0, broadcast_1.broadcastLog)(cronName, "No currencies in Redis, fetching from database");
        try {
            // Fetch currencies from database if not in Redis
            const currenciesFromDb = await db_1.models.currency.findAll({
                where: { status: true },
                attributes: ["id", "code"]
            });
            if (!currenciesFromDb || currenciesFromDb.length === 0) {
                (0, broadcast_1.broadcastLog)(cronName, "No currencies found in database, skipping rate update", "warning");
                return; // Exit gracefully instead of throwing
            }
            currencies = currenciesFromDb.map(c => ({
                id: c.code,
                code: c.code
            }));
            // Cache them for future use
            await redis.set("currencies", JSON.stringify(currencies), "EX", 86400); // 24 hours
            (0, broadcast_1.broadcastLog)(cronName, `Cached ${currencies.length} currencies from database`);
        }
        catch (dbError) {
            (0, broadcast_1.broadcastLog)(cronName, `Database fetch failed: ${dbError.message}`, "error");
            return; // Exit gracefully
        }
    }
    else {
        try {
            currencies = JSON.parse(currenciesRaw);
        }
        catch (parseError) {
            (0, broadcast_1.broadcastLog)(cronName, `Error parsing currencies data: ${parseError.message}`, "error");
            return; // Exit gracefully instead of throwing
        }
        if (!Array.isArray(currencies)) {
            (0, broadcast_1.broadcastLog)(cronName, "Currencies data is not an array", "error");
            return; // Exit gracefully
        }
    }
    for (const currency of currencies) {
        if (Object.prototype.hasOwnProperty.call(exchangeRates, currency.id)) {
            ratesToUpdate[currency.id] = exchangeRates[currency.id];
        }
    }
    (0, broadcast_1.broadcastLog)(cronName, `Updating rates for ${Object.keys(ratesToUpdate).length} currencies`);
    await (0, utils_1.updateCurrencyRates)(ratesToUpdate);
    (0, broadcast_1.broadcastLog)(cronName, "Currency rates updated in database", "success");
    await (0, utils_1.cacheCurrencies)();
    (0, broadcast_1.broadcastLog)(cronName, "Currencies cached successfully", "success");
}
async function cacheExchangeCurrencies() {
    const cronName = "cacheExchangeCurrencies";
    (0, broadcast_1.broadcastLog)(cronName, "Caching exchange currencies");
    const currencies = await (0, index_get_1.getCurrencies)();
    await redis.set("exchangeCurrencies", JSON.stringify(currencies), "EX", 1800);
    (0, broadcast_1.broadcastLog)(cronName, "Exchange currencies cached", "success");
}
async function processCurrenciesPrices() {
    const cronName = "processCurrenciesPrices";
    (0, broadcast_1.broadcastLog)(cronName, "Starting processCurrenciesPrices");
    let unblockTime = await (0, utils_2.loadBanStatus)();
    try {
        if (Date.now() < unblockTime) {
            const waitTime = unblockTime - Date.now();
            console.log(`Waiting for ${(0, utils_2.formatWaitTime)(waitTime)} until unblock time`);
            (0, broadcast_1.broadcastLog)(cronName, `Currently banned; waiting for ${(0, utils_2.formatWaitTime)(waitTime)}`, "info");
            return; // Exit if currently banned
        }
        const exchange = await exchange_1.default.startExchange();
        if (!exchange) {
            (0, broadcast_1.broadcastLog)(cronName, "Exchange instance not available; exiting", "error");
            return;
        }
        let marketsCache = [];
        let currenciesCache = [];
        try {
            marketsCache = await db_1.models.exchangeMarket.findAll({
                where: { status: true },
                attributes: ["currency", "pair"],
            });
            (0, broadcast_1.broadcastLog)(cronName, `Fetched ${marketsCache.length} active market records`);
        }
        catch (err) {
            (0, logger_1.logError)("processCurrenciesPrices - fetch markets", err, __filename);
            (0, broadcast_1.broadcastLog)(cronName, `Error fetching market records: ${err.message}`, "error");
            throw err;
        }
        try {
            currenciesCache = await db_1.models.exchangeCurrency.findAll({
                attributes: ["currency", "id", "price", "status"],
            });
            (0, broadcast_1.broadcastLog)(cronName, `Fetched ${currenciesCache.length} exchange currency records`);
        }
        catch (err) {
            (0, logger_1.logError)("processCurrenciesPrices - fetch currencies", err, __filename);
            (0, broadcast_1.broadcastLog)(cronName, `Error fetching currencies: ${err.message}`, "error");
            throw err;
        }
        const marketSymbols = marketsCache.map((market) => `${market.currency}/${market.pair}`);
        if (!marketSymbols.length) {
            const error = new Error("No market symbols found");
            (0, logger_1.logError)("processCurrenciesPrices - market symbols", error, __filename);
            (0, broadcast_1.broadcastLog)(cronName, error.message, "error");
            throw error;
        }
        (0, broadcast_1.broadcastLog)(cronName, `Market symbols: ${marketSymbols.join(", ")}`);
        let markets = {};
        try {
            if (exchange.has["fetchLastPrices"]) {
                markets = await exchange.fetchLastPrices(marketSymbols);
            }
            else {
                markets = await exchange.fetchTickers(marketSymbols);
            }
            (0, broadcast_1.broadcastLog)(cronName, "Fetched market data from exchange");
        }
        catch (error) {
            const result = await (0, utils_2.handleExchangeError)(error, exchange_1.default);
            if (typeof result === "number") {
                unblockTime = result;
                await (0, utils_2.saveBanStatus)(unblockTime);
                console.log(`Ban detected. Blocked until ${new Date(unblockTime).toLocaleString()}`);
                (0, broadcast_1.broadcastLog)(cronName, `Ban detected. Blocked until ${new Date(unblockTime).toLocaleString()}`, "error");
                return;
            }
            (0, logger_1.logError)("processCurrenciesPrices - fetch markets data", error, __filename);
            (0, broadcast_1.broadcastLog)(cronName, `Error fetching market data: ${error.message}`, "error");
            throw error;
        }
        const usdtPairs = Object.keys(markets).filter((symbol) => symbol.endsWith("/USDT"));
        (0, broadcast_1.broadcastLog)(cronName, `Found ${usdtPairs.length} USDT pairs in market data`);
        const bulkUpdateData = usdtPairs
            .map((symbol) => {
            const currency = symbol.split("/")[0];
            const market = markets[symbol];
            let price;
            if (exchange.has["fetchLastPrices"]) {
                price = market.price;
            }
            else {
                price = market.last;
            }
            if (!price || isNaN(parseFloat(price))) {
                console.warn(`Invalid or missing price for symbol: ${symbol}, market data: ${JSON.stringify(market)}`);
                (0, broadcast_1.broadcastLog)(cronName, `Invalid or missing price for symbol: ${symbol}`, "warning");
                return null;
            }
            const matchingCurrency = currenciesCache.find((dbCurrency) => dbCurrency.currency === currency);
            if (matchingCurrency) {
                matchingCurrency.price = parseFloat(price);
                return matchingCurrency;
            }
            return null;
        })
            .filter((item) => item !== null);
        const usdtCurrency = currenciesCache.find((dbCurrency) => dbCurrency.currency === "USDT");
        if (usdtCurrency) {
            usdtCurrency.price = 1;
            bulkUpdateData.push(usdtCurrency);
        }
        (0, broadcast_1.broadcastLog)(cronName, `Prepared bulk update data for ${bulkUpdateData.length} currencies`);
        try {
            await db_1.sequelize.transaction(async (transaction) => {
                for (const item of bulkUpdateData) {
                    await item.save({ transaction });
                }
            });
            (0, broadcast_1.broadcastLog)(cronName, "Bulk update of currency prices completed", "success");
        }
        catch (error) {
            (0, logger_1.logError)("processCurrenciesPrices - update database", error, __filename);
            (0, broadcast_1.broadcastLog)(cronName, `Error updating database: ${error.message}`, "error");
            throw error;
        }
    }
    catch (error) {
        (0, logger_1.logError)("processCurrenciesPrices", error, __filename);
        (0, broadcast_1.broadcastLog)(cronName, `processCurrenciesPrices failed: ${error.message}`, "error");
        throw error;
    }
}
async function updateCurrencyPricesBulk(data) {
    const cronName = "updateCurrencyPricesBulk";
    (0, broadcast_1.broadcastLog)(cronName, `Starting bulk update for ${data.length} currency prices`);
    try {
        await db_1.sequelize.transaction(async (transaction) => {
            for (const item of data) {
                await db_1.models.exchangeCurrency.update({ price: item.price }, { where: { id: item.id }, transaction });
            }
        });
        (0, broadcast_1.broadcastLog)(cronName, "Bulk update of currency prices succeeded", "success");
    }
    catch (error) {
        (0, logger_1.logError)("updateCurrencyPricesBulk", error, __filename);
        (0, broadcast_1.broadcastLog)(cronName, `Bulk update failed: ${error.message}`, "error");
        throw error;
    }
}
exports.default = CronJobManager;
