"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processForexInvestments = processForexInvestments;
exports.getActiveForexInvestments = getActiveForexInvestments;
exports.processForexInvestment = processForexInvestment;
const db_1 = require("@b/db");
const logger_1 = require("../logger");
const date_fns_1 = require("date-fns");
const emails_1 = require("../emails");
const notifications_1 = require("../notifications");
const affiliate_1 = require("../affiliate");
const broadcast_1 = require("./broadcast");
const logger_2 = require("../logger");
// Forex Cron: processForexInvestments runs periodically.
async function processForexInvestments() {
    const cronName = "processForexInvestments";
    const startTime = Date.now();
    try {
        (0, broadcast_1.broadcastStatus)(cronName, "running");
        (0, broadcast_1.broadcastLog)(cronName, "Starting Forex investments processing");
        const activeInvestments = await getActiveForexInvestments();
        const total = activeInvestments.length;
        (0, broadcast_1.broadcastLog)(cronName, `Found ${total} active forex investments`);
        for (let i = 0; i < total; i++) {
            const investment = activeInvestments[i];
            (0, broadcast_1.broadcastLog)(cronName, `Processing forex investment id ${investment.id} (current status: ${investment.status})`);
            try {
                const updated = await processForexInvestment(investment);
                if (updated) {
                    (0, broadcast_1.broadcastLog)(cronName, `Successfully processed forex investment id ${investment.id}`, "success");
                }
                else {
                    (0, broadcast_1.broadcastLog)(cronName, `No update for forex investment id ${investment.id}`, "warning");
                }
            }
            catch (error) {
                (0, logger_1.logError)(`processForexInvestments - investment ${investment.id}`, error, __filename);
                (0, broadcast_1.broadcastLog)(cronName, `Error processing forex investment id ${investment.id}: ${error.message}`, "error");
                continue;
            }
            const progress = Math.round(((i + 1) / total) * 100);
            (0, broadcast_1.broadcastProgress)(cronName, progress);
        }
        (0, broadcast_1.broadcastStatus)(cronName, "completed", {
            duration: Date.now() - startTime,
        });
        (0, broadcast_1.broadcastLog)(cronName, "Forex investments processing completed", "success");
    }
    catch (error) {
        (0, logger_1.logError)("processForexInvestments", error, __filename);
        (0, broadcast_1.broadcastStatus)(cronName, "failed");
        (0, broadcast_1.broadcastLog)(cronName, `Forex investments processing failed: ${error.message}`, "error");
        throw error;
    }
}
async function getActiveForexInvestments() {
    try {
        return await db_1.models.forexInvestment.findAll({
            where: {
                status: "ACTIVE",
            },
            include: [
                {
                    model: db_1.models.forexPlan,
                    as: "plan",
                    attributes: [
                        "id",
                        "name",
                        "title",
                        "description",
                        "defaultProfit",
                        "defaultResult",
                        "currency",
                        "walletType",
                    ],
                },
                {
                    model: db_1.models.forexDuration,
                    as: "duration",
                    attributes: ["id", "duration", "timeframe"],
                },
            ],
            order: [
                ["status", "ASC"],
                ["createdAt", "ASC"],
            ],
        });
    }
    catch (error) {
        (0, logger_1.logError)("getActiveForexInvestments", error, __filename);
        throw error;
    }
}
async function processForexInvestment(investment, retryCount = 0) {
    const cronName = "processForexInvestments";
    const maxRetries = 3;
    try {
        if (investment.status === "COMPLETED") {
            (0, broadcast_1.broadcastLog)(cronName, `Investment ${investment.id} is already COMPLETED; skipping`, "info");
            return null;
        }
        (0, broadcast_1.broadcastLog)(cronName, `Fetching user for investment ${investment.id}`);
        const user = await fetchUser(investment.userId);
        if (!user) {
            (0, broadcast_1.broadcastLog)(cronName, `User not found for investment ${investment.id}`, "error");
            return null;
        }
        const roi = calculateRoi(investment);
        (0, broadcast_1.broadcastLog)(cronName, `Calculated ROI (${roi}) for investment ${investment.id}`);
        const investmentResult = determineInvestmentResult(investment);
        (0, broadcast_1.broadcastLog)(cronName, `Determined result (${investmentResult}) for investment ${investment.id}`);
        if (shouldProcessInvestment(investment, roi, investmentResult)) {
            (0, broadcast_1.broadcastLog)(cronName, `Investment ${investment.id} is eligible for processing (end date passed)`);
            const updatedInvestment = await handleInvestmentUpdate(investment, user, roi, investmentResult);
            if (updatedInvestment) {
                await postProcessInvestment(user, investment, updatedInvestment);
            }
            return updatedInvestment;
        }
        else {
            (0, broadcast_1.broadcastLog)(cronName, `Investment ${investment.id} is not ready for processing (end date not reached)`, "info");
            return null;
        }
    }
    catch (error) {
        (0, logger_1.logError)(`processForexInvestment - General`, error, __filename);
        (0, broadcast_1.broadcastLog)(cronName, `Error processing investment ${investment.id}: ${error.message}`, "error");
        // Retry logic
        if (retryCount < maxRetries) {
            (0, broadcast_1.broadcastLog)(cronName, `Retrying investment ${investment.id} (attempt ${retryCount + 1}/${maxRetries})`, "warning");
            // Exponential backoff: wait 2^retryCount seconds
            await new Promise(resolve => setTimeout(resolve, Math.pow(2, retryCount) * 1000));
            // Retry with incremented count
            return processForexInvestment(investment, retryCount + 1);
        }
        else {
            // Max retries reached, mark investment for manual review
            try {
                await db_1.models.forexInvestment.update({
                    status: "CANCELLED",
                    metadata: JSON.stringify({
                        error: error.message,
                        failedAt: new Date().toISOString(),
                        retries: retryCount
                    })
                }, { where: { id: investment.id } });
                (0, broadcast_1.broadcastLog)(cronName, `Investment ${investment.id} marked as CANCELLED after ${maxRetries} retries`, "error");
                // Create notification for admin
                await (0, notifications_1.createNotification)({
                    userId: investment.userId,
                    relatedId: investment.id,
                    title: "Forex Investment Processing Failed",
                    message: `Investment ${investment.id} failed to process after ${maxRetries} attempts. Manual review required.`,
                    type: "system",
                    link: `/admin/forex/investment/${investment.id}`,
                });
            }
            catch (updateError) {
                (0, logger_1.logError)(`processForexInvestment - Failed to mark as cancelled`, updateError, __filename);
            }
        }
        throw error;
    }
}
async function fetchUser(userId) {
    try {
        const user = await db_1.models.user.findByPk(userId);
        if (!user) {
            (0, logger_1.logError)(`fetchUser`, new Error(`User not found: ${userId}`), __filename);
        }
        return user;
    }
    catch (error) {
        (0, logger_1.logError)(`fetchUser`, error, __filename);
        throw error;
    }
}
function calculateRoi(investment) {
    var _a;
    const roi = (_a = investment.profit) !== null && _a !== void 0 ? _a : investment.plan.defaultProfit;
    return roi;
}
function determineInvestmentResult(investment) {
    const result = investment.result || investment.plan.defaultResult;
    return result;
}
function shouldProcessInvestment(investment, roi, investmentResult) {
    const endDate = calculateEndDate(investment);
    return (0, date_fns_1.isPast)(endDate);
}
function calculateEndDate(investment) {
    const createdAt = new Date(investment.createdAt);
    let endDate;
    switch (investment.duration.timeframe) {
        case "HOUR":
            endDate = (0, date_fns_1.addHours)(createdAt, investment.duration.duration);
            break;
        case "DAY":
            endDate = (0, date_fns_1.addDays)(createdAt, investment.duration.duration);
            break;
        case "WEEK":
            endDate = (0, date_fns_1.addDays)(createdAt, investment.duration.duration * 7);
            break;
        case "MONTH":
            endDate = (0, date_fns_1.addDays)(createdAt, investment.duration.duration * 30);
            break;
        default:
            endDate = (0, date_fns_1.addHours)(createdAt, investment.duration.duration);
            break;
    }
    return endDate;
}
async function handleInvestmentUpdate(investment, user, roi, investmentResult) {
    var _a;
    const cronName = "processForexInvestments";
    let updatedInvestment;
    // Use a single transaction for all updates
    const t = await db_1.sequelize.transaction();
    try {
        (0, broadcast_1.broadcastLog)(cronName, `Starting update for investment ${investment.id}`);
        const wallet = await fetchWallet(user.id, investment.plan.currency, investment.plan.walletType, t);
        if (!wallet) {
            (0, broadcast_1.broadcastLog)(cronName, `Wallet not found for user ${user.id} (investment ${investment.id})`, "error");
            await t.rollback();
            return null;
        }
        const amount = (_a = investment.amount) !== null && _a !== void 0 ? _a : 0;
        const newBalance = wallet.balance;
        (0, broadcast_1.broadcastLog)(cronName, `Fetched wallet. Current balance: ${newBalance}, investment amount: ${amount}`);
        if (investmentResult === "WIN") {
            await db_1.models.wallet.update({ balance: newBalance + amount + roi }, { where: { id: wallet.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Wallet updated for WIN case. New balance: ${newBalance + amount + roi}`);
            await db_1.models.transaction.create({
                userId: wallet.userId,
                walletId: wallet.id,
                amount: roi,
                description: `Investment ROI: Plan "${investment.plan.title}" | Duration: ${investment.duration.duration} ${investment.duration.timeframe}`,
                status: "COMPLETED",
                type: "FOREX_INVESTMENT_ROI",
            }, { transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Transaction record created for WIN case for investment ${investment.id}`);
            await db_1.models.forexInvestment.update({ status: "COMPLETED", result: investmentResult, profit: roi }, { where: { id: investment.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Forex investment ${investment.id} updated to COMPLETED (WIN)`);
            // Log the investment completion
            (0, logger_2.logInfo)("forex-investment-completion", `Forex investment ${investment.id} completed for user ${user.id} with result: ${investmentResult}, ROI: ${roi}`, __filename);
        }
        else if (investmentResult === "LOSS") {
            // In LOSS case, roi represents the loss amount (negative value)
            // Return the remaining amount after deducting the loss
            const remainingAmount = Math.max(0, amount - Math.abs(roi));
            await db_1.models.wallet.update({ balance: newBalance + remainingAmount }, { where: { id: wallet.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Wallet updated for LOSS case. New balance: ${newBalance + remainingAmount}`);
            await db_1.models.transaction.create({
                userId: wallet.userId,
                walletId: wallet.id,
                amount: -Math.abs(roi),
                description: `Investment ROI: Plan "${investment.plan.title}" | Duration: ${investment.duration.duration} ${investment.duration.timeframe}`,
                status: "COMPLETED",
                type: "FOREX_INVESTMENT_ROI",
            }, { transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Transaction record created for LOSS case for investment ${investment.id}`);
            await db_1.models.forexInvestment.update({ status: "COMPLETED", result: investmentResult, profit: roi }, { where: { id: investment.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Forex investment ${investment.id} updated to COMPLETED (LOSS)`);
            // Log the investment completion
            (0, logger_2.logInfo)("forex-investment-completion", `Forex investment ${investment.id} completed for user ${user.id} with result: ${investmentResult}, Loss: ${-Math.abs(roi)}`, __filename);
        }
        else {
            // For DRAW or other cases
            await db_1.models.wallet.update({ balance: newBalance + amount }, { where: { id: wallet.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Wallet updated for DRAW case. New balance: ${newBalance + amount}`);
            await db_1.models.transaction.create({
                userId: wallet.userId,
                walletId: wallet.id,
                amount: 0,
                description: `Investment ROI: Plan "${investment.plan.title}" | Duration: ${investment.duration.duration} ${investment.duration.timeframe}`,
                status: "COMPLETED",
                type: "FOREX_INVESTMENT_ROI",
            }, { transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Transaction record created for DRAW case for investment ${investment.id}`);
            await db_1.models.forexInvestment.update({ status: "COMPLETED", result: investmentResult, profit: roi }, { where: { id: investment.id }, transaction: t });
            (0, broadcast_1.broadcastLog)(cronName, `Forex investment ${investment.id} updated to COMPLETED (DRAW)`);
            // Log the investment completion
            (0, logger_2.logInfo)("forex-investment-completion", `Forex investment ${investment.id} completed for user ${user.id} with result: ${investmentResult}, No gain or loss`, __filename);
        }
        updatedInvestment = await db_1.models.forexInvestment.findByPk(investment.id, {
            include: [
                { model: db_1.models.forexPlan, as: "plan" },
                { model: db_1.models.forexDuration, as: "duration" },
            ],
            transaction: t,
        });
        await t.commit();
        (0, broadcast_1.broadcastLog)(cronName, `Transaction committed for investment ${investment.id}`, "success");
    }
    catch (error) {
        await t.rollback();
        (0, broadcast_1.broadcastLog)(cronName, `Error updating investment ${investment.id}: ${error.message}`, "error");
        (0, logger_1.logError)(`handleInvestmentUpdate`, error, __filename);
        return null;
    }
    return updatedInvestment;
}
async function fetchWallet(userId, currency, walletType, transaction) {
    try {
        const wallet = await db_1.models.wallet.findOne({
            where: { userId, currency, type: walletType },
            transaction,
        });
        if (!wallet)
            throw new Error("Wallet not found");
        return wallet;
    }
    catch (error) {
        (0, logger_1.logError)(`fetchWallet`, error, __filename);
        throw error;
    }
}
async function postProcessInvestment(user, investment, updatedInvestment) {
    var _a;
    const cronName = "processForexInvestments";
    try {
        (0, broadcast_1.broadcastLog)(cronName, `Sending investment email for investment ${investment.id}`);
        await (0, emails_1.sendInvestmentEmail)(user, investment.plan, investment.duration, updatedInvestment, "ForexInvestmentCompleted");
        (0, broadcast_1.broadcastLog)(cronName, `Investment email sent for investment ${investment.id}`, "success");
        (0, broadcast_1.broadcastLog)(cronName, `Creating notification for investment ${investment.id}`);
        await (0, notifications_1.createNotification)({
            userId: user.id,
            relatedId: updatedInvestment.id,
            title: "Forex Investment Completed",
            message: `Your Forex investment of ${investment.amount} ${investment.plan.currency} has been completed with a status of ${updatedInvestment.result}`,
            type: "system",
            link: `/forex/investments/${updatedInvestment.id}`,
            actions: [
                {
                    label: "View Investment",
                    link: `/forex/investments/${updatedInvestment.id}`,
                    primary: true,
                },
            ],
        });
        (0, broadcast_1.broadcastLog)(cronName, `Notification created for investment ${investment.id}`, "success");
        (0, broadcast_1.broadcastLog)(cronName, `Processing rewards for investment ${investment.id}`);
        await (0, affiliate_1.processRewards)(user.id, (_a = investment.amount) !== null && _a !== void 0 ? _a : 0, "FOREX_INVESTMENT", investment.plan.currency);
        (0, broadcast_1.broadcastLog)(cronName, `Rewards processed for investment ${investment.id}`, "success");
    }
    catch (error) {
        (0, broadcast_1.broadcastLog)(cronName, `Error in postProcessInvestment for ${investment.id}: ${error.message}`, "error");
        (0, logger_1.logError)(`postProcessInvestment`, error, __filename);
    }
}
