"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processIcoOfferings = processIcoOfferings;
const db_1 = require("@b/db");
const sequelize_1 = require("sequelize");
const logger_1 = require("../logger");
const broadcast_1 = require("./broadcast");
async function processIcoOfferings() {
    const cronName = "processIcoOfferings";
    const startTime = Date.now();
    try {
        (0, broadcast_1.broadcastStatus)(cronName, "running");
        (0, broadcast_1.broadcastLog)(cronName, "Starting ICO offerings processing");
        // Fetch ICO offerings that are either UPCOMING or ACTIVE.
        // We only want to check offerings that might need a status change.
        const offerings = await db_1.models.icoTokenOffering.findAll({
            where: {
                status: { [sequelize_1.Op.in]: ["UPCOMING", "ACTIVE"] },
            },
        });
        (0, broadcast_1.broadcastLog)(cronName, `Found ${offerings.length} ICO offerings to evaluate`, "info");
        const currentDate = new Date();
        for (const offering of offerings) {
            try {
                // If offering is UPCOMING and the startDate has passed, change to ACTIVE.
                if (offering.status === "UPCOMING" &&
                    offering.startDate &&
                    currentDate >= offering.startDate) {
                    await offering.update({ status: "ACTIVE" });
                    (0, broadcast_1.broadcastLog)(cronName, `Offering ${offering.id} changed from UPCOMING to ACTIVE`, "success");
                }
                // If offering is ACTIVE and the endDate has passed, change to SUCCESS.
                else if (offering.status === "ACTIVE" &&
                    offering.endDate &&
                    currentDate >= offering.endDate) {
                    await offering.update({ status: "SUCCESS" });
                    (0, broadcast_1.broadcastLog)(cronName, `Offering ${offering.id} changed from ACTIVE to SUCCESS`, "success");
                }
                else {
                    (0, broadcast_1.broadcastLog)(cronName, `Offering ${offering.id} not eligible for update (status: ${offering.status}, startDate: ${offering.startDate}, endDate: ${offering.endDate})`, "info");
                }
            }
            catch (error) {
                (0, logger_1.logError)(`processIcoOfferings - offering ${offering.id}`, error, __filename);
                (0, broadcast_1.broadcastLog)(cronName, `Error updating offering ${offering.id}: ${error.message}`, "error");
            }
        }
        (0, broadcast_1.broadcastStatus)(cronName, "completed", {
            duration: Date.now() - startTime,
        });
        (0, broadcast_1.broadcastLog)(cronName, "ICO offerings processing completed", "success");
    }
    catch (error) {
        (0, logger_1.logError)("processIcoOfferings", error, __filename);
        (0, broadcast_1.broadcastStatus)(cronName, "failed");
        (0, broadcast_1.broadcastLog)(cronName, `ICO offerings processing failed: ${error.message}`, "error");
        throw error;
    }
}
