"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.processNFTBackups = processNFTBackups;
exports.initializeNFTBackupSchedules = initializeNFTBackupSchedules;
exports.stopNFTBackupSchedules = stopNFTBackupSchedules;
exports.createNFTBackupSchedule = createNFTBackupSchedule;
exports.deleteNFTBackupSchedule = deleteNFTBackupSchedule;
const db_1 = require("@b/db");
const logger_1 = require("../logger");
const broadcast_1 = require("./broadcast");
// Store active backup schedules
const activeSchedules = new Map();
/**
 * Dynamically get the blockchain backup service if available
 */
async function getBackupService(chain) {
    try {
        // @ts-ignore - Dynamic import for optional NFT extension
        const backupModule = await Promise.resolve().then(() => __importStar(require("@b/api/(ext)/nft/utils/blockchain-backup-service")));
        return await backupModule.getBlockchainBackupService(chain);
    }
    catch (error) {
        (0, broadcast_1.broadcastLog)("NFT Backup", `Backup service not available for chain: ${chain}`);
        return null;
    }
}
/**
 * Process NFT blockchain backups based on configured schedules
 */
async function processNFTBackups() {
    var _a, _b;
    try {
        (0, broadcast_1.broadcastStatus)("NFT Backup", "running", { message: "Processing scheduled NFT backups" });
        // Get all configured backup schedules
        const schedules = await ((_a = db_1.models.systemSetting) === null || _a === void 0 ? void 0 : _a.findAll({
            where: {
                key: {
                    [db_1.models.Sequelize.Op.like]: 'nft_backup_schedule_%'
                }
            }
        }));
        if (!schedules || schedules.length === 0) {
            (0, broadcast_1.broadcastLog)("NFT Backup", "No backup schedules configured");
            return;
        }
        for (const schedule of schedules) {
            try {
                const config = JSON.parse(schedule.value);
                // Check if backup is enabled
                if (!config.enabled) {
                    continue;
                }
                // Check if it's time to run based on schedule
                const lastRun = config.lastRun ? new Date(config.lastRun) : null;
                const nextRun = config.nextRun ? new Date(config.nextRun) : null;
                const now = new Date();
                if (nextRun && now >= nextRun) {
                    (0, broadcast_1.broadcastLog)("NFT Backup", `Running backup for chain: ${config.chain}`);
                    // Get backup service
                    const backupService = await getBackupService(config.chain);
                    if (!backupService) {
                        continue;
                    }
                    // Execute backup based on type
                    if (config.backupType === "FULL") {
                        await backupService.createBackup(config.includeDisputes);
                        (0, broadcast_1.broadcastLog)("NFT Backup", `Full backup completed for ${config.chain}`);
                    }
                    else {
                        await backupService.createIncrementalBackup();
                        (0, broadcast_1.broadcastLog)("NFT Backup", `Incremental backup completed for ${config.chain}`);
                    }
                    // Update last run and calculate next run
                    config.lastRun = now.toISOString();
                    config.nextRun = calculateNextRun(config.schedule, now).toISOString();
                    // Save updated schedule
                    await ((_b = db_1.models.systemSetting) === null || _b === void 0 ? void 0 : _b.update({ value: JSON.stringify(config) }, { where: { key: schedule.key } }));
                }
            }
            catch (error) {
                (0, logger_1.logError)("process_nft_backup_schedule", error, __filename);
                (0, broadcast_1.broadcastLog)("NFT Backup", `Error processing backup for schedule: ${error.message}`);
            }
        }
        (0, broadcast_1.broadcastStatus)("NFT Backup", "completed", { message: "All scheduled backups processed" });
    }
    catch (error) {
        (0, logger_1.logError)("process_nft_backups", error, __filename);
        (0, broadcast_1.broadcastStatus)("NFT Backup", "failed", { error: error.message });
    }
}
/**
 * Initialize NFT backup schedules from database on startup
 */
async function initializeNFTBackupSchedules() {
    var _a;
    try {
        const schedules = await ((_a = db_1.models.systemSetting) === null || _a === void 0 ? void 0 : _a.findAll({
            where: {
                key: {
                    [db_1.models.Sequelize.Op.like]: 'nft_backup_schedule_%'
                }
            }
        }));
        if (!schedules || schedules.length === 0) {
            return;
        }
        for (const schedule of schedules) {
            try {
                const config = JSON.parse(schedule.value);
                if (!config.enabled) {
                    continue;
                }
                // Setup interval for this schedule
                const intervalMs = getIntervalFromSchedule(config.schedule);
                if (intervalMs > 0) {
                    const intervalId = setInterval(async () => {
                        try {
                            const backupService = await getBackupService(config.chain);
                            if (!backupService) {
                                return;
                            }
                            if (config.backupType === "FULL") {
                                await backupService.createBackup(config.includeDisputes);
                            }
                            else {
                                await backupService.createIncrementalBackup();
                            }
                            (0, broadcast_1.broadcastLog)("NFT Backup", `Backup completed for ${config.chain}`);
                        }
                        catch (error) {
                            (0, logger_1.logError)("nft_backup_interval", error, __filename);
                        }
                    }, intervalMs);
                    activeSchedules.set(`nft-backup-${config.chain}`, intervalId);
                    console.log(`[NFT Backup] Initialized schedule for ${config.chain}: ${config.schedule}`);
                }
            }
            catch (error) {
                (0, logger_1.logError)("init_nft_backup_schedule", error, __filename);
            }
        }
    }
    catch (error) {
        (0, logger_1.logError)("initialize_nft_backup_schedules", error, __filename);
    }
}
/**
 * Stop all active NFT backup schedules
 */
function stopNFTBackupSchedules() {
    for (const [key, intervalId] of activeSchedules) {
        clearInterval(intervalId);
        activeSchedules.delete(key);
    }
    console.log("[NFT Backup] All backup schedules stopped");
}
/**
 * Calculate next run time based on schedule type
 */
function calculateNextRun(schedule, fromDate = new Date()) {
    const nextRun = new Date(fromDate);
    switch (schedule) {
        case "HOURLY":
            nextRun.setHours(nextRun.getHours() + 1);
            break;
        case "DAILY":
            nextRun.setDate(nextRun.getDate() + 1);
            break;
        case "WEEKLY":
            nextRun.setDate(nextRun.getDate() + 7);
            break;
        case "MONTHLY":
            nextRun.setMonth(nextRun.getMonth() + 1);
            break;
        default:
            // Default to daily
            nextRun.setDate(nextRun.getDate() + 1);
    }
    return nextRun;
}
/**
 * Get interval in milliseconds from schedule type
 */
function getIntervalFromSchedule(schedule) {
    switch (schedule) {
        case "HOURLY":
            return 60 * 60 * 1000; // 1 hour
        case "DAILY":
            return 24 * 60 * 60 * 1000; // 24 hours
        case "WEEKLY":
            return 7 * 24 * 60 * 60 * 1000; // 7 days
        case "MONTHLY":
            return 30 * 24 * 60 * 60 * 1000; // 30 days (approximate)
        default:
            return 24 * 60 * 60 * 1000; // Default to daily
    }
}
/**
 * Create or update NFT backup schedule
 */
async function createNFTBackupSchedule(chain, schedule, backupType, includeDisputes = false, enabled = true) {
    var _a;
    try {
        const key = `nft_backup_schedule_${chain}`;
        // Stop existing schedule if any
        const existingId = activeSchedules.get(`nft-backup-${chain}`);
        if (existingId) {
            clearInterval(existingId);
            activeSchedules.delete(`nft-backup-${chain}`);
        }
        // Create new schedule configuration
        const config = {
            chain,
            schedule,
            backupType,
            includeDisputes,
            enabled,
            lastRun: null,
            nextRun: calculateNextRun(schedule).toISOString(),
            createdAt: new Date().toISOString()
        };
        // Save to database
        await ((_a = db_1.models.systemSetting) === null || _a === void 0 ? void 0 : _a.upsert({
            key,
            value: JSON.stringify(config)
        }));
        // Setup new interval if enabled
        if (enabled) {
            const intervalMs = getIntervalFromSchedule(schedule);
            if (intervalMs > 0) {
                const intervalId = setInterval(async () => {
                    var _a;
                    try {
                        const backupService = await getBackupService(chain);
                        if (!backupService) {
                            return;
                        }
                        if (backupType === "FULL") {
                            await backupService.createBackup(includeDisputes);
                        }
                        else {
                            await backupService.createIncrementalBackup();
                        }
                        // Update last run time
                        config.lastRun = new Date().toISOString();
                        config.nextRun = calculateNextRun(schedule).toISOString();
                        await ((_a = db_1.models.systemSetting) === null || _a === void 0 ? void 0 : _a.update({ value: JSON.stringify(config) }, { where: { key } }));
                        (0, broadcast_1.broadcastLog)("NFT Backup", `Backup completed for ${chain}`);
                    }
                    catch (error) {
                        (0, logger_1.logError)("nft_backup_schedule_execution", error, __filename);
                    }
                }, intervalMs);
                activeSchedules.set(`nft-backup-${chain}`, intervalId);
            }
        }
        return config;
    }
    catch (error) {
        (0, logger_1.logError)("create_nft_backup_schedule", error, __filename);
        throw error;
    }
}
/**
 * Delete NFT backup schedule
 */
async function deleteNFTBackupSchedule(chain) {
    var _a;
    try {
        // Stop active schedule
        const scheduleId = activeSchedules.get(`nft-backup-${chain}`);
        if (scheduleId) {
            clearInterval(scheduleId);
            activeSchedules.delete(`nft-backup-${chain}`);
        }
        // Delete from database
        await ((_a = db_1.models.systemSetting) === null || _a === void 0 ? void 0 : _a.destroy({
            where: { key: `nft_backup_schedule_${chain}` }
        }));
        (0, broadcast_1.broadcastLog)("NFT Backup", `Backup schedule deleted for ${chain}`);
    }
    catch (error) {
        (0, logger_1.logError)("delete_nft_backup_schedule", error, __filename);
        throw error;
    }
}
